#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import evas
import ecore
import edje
import edje.decorators
import logging

from terra.ui.screen import Screen
from album_view import AlbumView

__all__ = ("AlbumArtScreen", )

log = logging.getLogger("canola.ui.album_art")

class AlbumArtScreen(Screen):
    view_name = "album_view"

    def __init__(self, canvas, main_window):
        Screen.__init__(self, canvas, "album_view", main_window, "Cover arts")
        self.main_window = main_window
        self.animating = False
        self.right = False

        # callbacks
        self.started_next_move = None
        self.started_prev_move = None
        self.finished_prev_move = None
        self.finished_next_move = None

        self.prev_aux_album = None
        self.prev_album = None
        self.current_album = None
        self.next_album = None
        self.next_aux_album = None

        self._setup_all_widgets()

    def _setup_all_widgets(self):
        self.w_prev_aux = self.part_swallow_get("album_view/aux_prev")
        self.w_prev = self.part_swallow_get("album_view/prev")
        self.w_current = self.part_swallow_get("album_view/current")
        self.w_next = self.part_swallow_get("album_view/next")
        self.w_next_aux = self.part_swallow_get("album_view/aux_next")

    def theme_changed(self):
        Screen.theme_changed(self)
        self._setup_all_widgets()

        if self.prev_aux_album:
            self.prev_aux_album.set_cover_widget(self.w_prev_aux)
        self.prev_album.set_cover_widget(self.w_prev)
        self.current_album.set_cover_widget(self.w_current)
        self.next_album.set_cover_widget(self.w_next)
        if self.next_aux_album:
            self.next_aux_album.set_cover_widget(self.w_next_aux)

        if self.current_name and self.current_artist:
            self.set_main_album_labels(self.current_name, self.current_artist)

        if self.prev_album:
            self.prev_album.theme_changed()

        if self.current_album:
            self.current_album.theme_changed()

        if self.next_album:
            self.next_album.theme_changed()
        self.w_prev.signal_emit("titles_clipper,hide,now", "")

    def setup_cover_widgets(self, cb_title_clicked, cb_cover_clicked):
        self.w_prev.signal_emit("titles_clipper,hide,now", "")
        self.prev_album = AlbumView(self.w_prev, self,
                                    cb_title_clicked, cb_cover_clicked)
        self.current_album = AlbumView(self.w_current, self,
                                       cb_title_clicked, cb_cover_clicked)
        self.next_album = AlbumView(self.w_next, self,
                                    cb_title_clicked, cb_cover_clicked)

        self.current_album.view.signal_callback_add("album_set,show", "*",
                                                    self._cb_cover_right)

    def _cb_cover_right(self, obj, emission, source):
        self.current_album.cb_cover_right()

    def set_main_album_labels(self, name, artist):
        try:
            obj = self.part_swallow_get("main_labels")
            obj.part_text_set("album_name", name)
            obj.part_text_set("album_artist", artist)
        except Exception, e:
            log.error("Could not setup labels: %s", e)

    def set_aux_album_labels(self, name, artist):
        try:
            obj = self.part_swallow_get("aux_labels")
            obj.part_text_set("album_name", name)
            obj.part_text_set("album_artist", artist)
        except Exception, e:
            log.error("Could not setup aux labels: %s", e)

    def setup_album_labels(self, prev_name, prev_artist,
                           name, artist, next_name, next_artist):
        self.prev_name = prev_name
        self.prev_artist = prev_artist
        self.current_name = name
        self.current_artist = artist
        self.next_name = next_name
        self.next_artist = next_artist
        self.set_main_album_labels(self.current_name, self.current_artist)

    def disable_cover_widget(self, widget):
        cover = widget.part_swallow_get("album_cover")
        cover.hide()
        widget.hide()

    def transition_from(self, old_view, end_callback=None):
        if self.current_album:
            if self.current_album.titles:
                for r in self.current_album.titles._list.renderers:
                    r.state_default()
        else:
            self.signal_emit("album_view,disable", "")
            labels = self.part_swallow_get("main_labels")
            labels.hide()
            self.disable_cover_widget(self.w_prev)
            self.disable_cover_widget(self.w_current)
            self.disable_cover_widget(self.w_next)
        Screen.transition_from(self, old_view, end_callback)

    def album_set_show(self):
        if not self.right:
            self.cover_click(self, None, None) # XXX: fix cover click method

    @edje.decorators.signal_callback("mouse,up,1",
                                     "album_view/current:album_clipper")
    def cover_click(obj, emission, source):
        def end(*ignore):
            obj.animating = False

        if obj.animating:
            return

        obj.animating = True
        if not obj.right:
            obj.right = True
            obj.w_current.signal_callback_add("album_set,show,finished", "", end)
            obj.w_current.signal_emit("album_set,show", "")
        else:
            obj.right = False
            obj.w_current.signal_callback_add("album_set,hide,finished", "", end)
            obj.w_current.signal_emit("album_set,hide", "")

    @edje.decorators.signal_callback("move,next", "")
    def move_next_cb(obj, emission, source):
        def end(*ignore):
            try:
                obj.w_current.signal_callback_del("album_set,hide,finished", "", end)
            except Exception:
                pass
            obj.set_aux_album_labels(obj.next_name, obj.next_artist)
            obj.w_current.signal_emit("titles_clipper,hide,left", "")
            obj.w_next.signal_emit("titles_clipper,show", "")
            obj.signal_emit("labels,next", "")
            obj.signal_emit("widgets,move,next", "")
            obj.started_next_move()

        if not obj.animating:
            obj.animating = True
            if obj.right:
                obj.w_current.signal_callback_add("album_set,hide,finished", "", end)
                obj.w_current.signal_emit("album_set,hide", "")
                obj.right = False
            else:
                end()

    @edje.decorators.signal_callback("finished,next", "*")
    def finished_next_cb(obj, emission, source):
        ecore.idler_add(obj.aux_finished_next)

    def aux_finished_next(self):
        self.w_current.signal_emit("titles_clipper,show,now", "")
        self.w_prev.signal_emit("titles_clipper,hide,now", "")
        if self.finished_next_move:
            self.finished_next_move()
        self.signal_emit("reset,1", "")
        return False

    @edje.decorators.signal_callback("move,prev", "")
    def move_prev_cb(obj, emission, source):
        def end(*ignore):
            try:
                obj.w_current.signal_callback_del("album_set,hide,finished", "", end)
            except Exception:
                pass
            obj.set_aux_album_labels(obj.prev_name, obj.prev_artist)
            obj.w_current.signal_emit("titles_clipper,hide,left", "")
            obj.w_prev.signal_emit("titles_clipper,show", "")
            obj.signal_emit("labels,prev", "")
            obj.signal_emit("widgets,move,prev", "")
            obj.started_prev_move()

        if not obj.animating:
            obj.animating = True
            if obj.right:
                obj.w_current.signal_callback_add("album_set,hide,finished", "", end)
                obj.w_current.signal_emit("album_set,hide", "")
                obj.right = False
            else:
                end()

    @edje.decorators.signal_callback("finished,prev", "*")
    def finished_prev_cb(obj, emission, source):
        ecore.idler_add(obj.aux_finished_prev)

    def aux_finished_prev(self):
        self.w_current.signal_emit("titles_clipper,show,now", "")
        self.w_prev.signal_emit("titles_clipper,hide,now", "")
        if self.finished_prev_move:
            self.finished_prev_move()
        self.signal_emit("reset,1", "")
        return False

    @edje.decorators.signal_callback("reset,finished", "")
    def reset_finished(obj, emission, source):
        obj.animating = False

    def handle_key_down(self, event):
        # Handle keys while focus is on canvas
        if event.keyname == "Left":
            if not self.animating:
                self.signal_emit("move,prev", "")
        elif event.keyname == "Right":
            if not self.animating:
                self.signal_emit("move,next", "")
        else:
            self._parent_widget.handle_key_down(event)
        return True

    @evas.decorators.del_callback
    def delete_cb(self):
        if self.prev_aux_album:
            self.prev_aux_album.delete()

        if self.prev_album:
            self.prev_album.delete()

        if self.current_album:
            self.current_album.delete()

        if self.next_album:
            self.next_album.delete()

        if self.next_aux_album:
            self.next_aux_album.delete()

