#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

from terra.core.model import Model


class MultitaskModel(Model):
    terra_type = "Model/Multitask"

    def __init__(self):
        Model.__init__(self, "Multitask")
        self.reset()
        self.callback_stop_player = None
        self.callback_player_added = None
        self.callback_player_removed = None

    def reset(self):
        self.tasks = {}     # maps tasks types to task controllers
        self.players = {}   # maps player types to task controllers

    def add(self, task_controller):
        terra_task_type = task_controller.task.terra_task_type
        assert terra_task_type not in self.tasks
        self.tasks[terra_task_type] = task_controller

    def remove(self, task_controller):
        terra_task_type = task_controller.task.terra_task_type
        del self.tasks[terra_task_type]

    def task_controller_for(self, task):
        try:
            return self.tasks[task.terra_task_type]
        except KeyError:
            return None

    def add_player(self, player_type, task_controller):
        assert len(self.players) == 0
        assert player_type not in self.players
        self.players[player_type] = task_controller

        if self.callback_player_added is not None:
            self.callback_player_added(task_controller.player)

    def remove_player(self, player_type, task_controller):
        assert len(self.players) == 1
        assert player_type in self.players
        assert self.players[player_type] is task_controller
        del self.players[player_type]

        if self.callback_player_removed is not None:
            self.callback_player_removed(task_controller.player)

    def stop_other_player(self, player_type):
        try:
            old = self.players[player_type]
        except KeyError:
            return

        self.remove_player(player_type, old)
        if self.callback_stop_player is not None:
            self.callback_stop_player(old)

    def get_player_task_controller(self):
        assert len(self.players) == 1
        return self.players.values()[0]
