#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import etk
import edje

from terra.core.terra_object import TerraObject


class ItemRenderer(etk.EdjeKineticRenderer, TerraObject):
    terra_type = "Renderer/EtkList/Item"

    def __init__(self, text_func=None, item_click=None, *a, **ka):
        if not callable(text_func):
            raise TypeError("text_func must be callable")

        etk.EdjeKineticRenderer.__init__(self)
        self.item_click = item_click
        self.text_func = text_func
        self.args = a
        self.kargs = ka

    def create_cell(self, canvas):
        cell = edje.Edje(canvas)
        etk.theme_edje_object_set(cell, self.list.theme_file,
                                  "item", "canola/conf")
        return cell

    def update_cell(self, cell, row):
        cell.part_text_set("label", self.text_func(row))

    def show_cell(self, cell, x, y, w, h):
        cell.geometry = x, y, w, h
        cell.show()

    def press(self, cell, part, row):
        cell.signal_emit("cell,state,selected", "")

    def release(self, cell, part, row):
        cell.signal_emit("cell,state,unselected", "")

    def click(self, cell, part, row):
        if self.item_click:
            self.item_click(row, self.list, *self.args, **self.kargs)


class ExpandButtonRenderer(etk.EdjeKineticRenderer, TerraObject):
    terra_type = "Renderer/EtkList/ExpandButton"

    def __init__(self, text_func=None, item_click=None, edit_click=None,
                 delete_click=None, bt_left_func=None, bt_right_func=None, *a, **ka):
        if text_func is None:
            raise ValueError("need a text func")

        self.bt_left_func = bt_left_func
        self.bt_right_func = bt_right_func

        etk.EdjeKineticRenderer.__init__(self)
        self.text_func = text_func
        self.item_click = item_click
        self.edit_click = edit_click
        self.delete_click = delete_click
        self.args = a
        self.kargs = ka
        self._open_list = []
        self._animating_list = []

    def create_cell(self, canvas):
        cell = edje.Edje(canvas)
        etk.theme_edje_object_set(cell, self.list.theme_file,
                                  "feed_item", "canola/conf")
        cell.signal_callback_add("options,open,finished", "*",
                                 self._end_open_animation)
        cell.signal_callback_add("options,close,finished", "*",
                                 self._end_close_animation)

        return cell

    def _end_open_animation(self, cell, signal, src):
        row = cell.data["row"]
        if row:
            self._animating_list.remove(row)
            self._open_list.append(row)

    def _end_close_animation(self, cell, signal, src):
        row = cell.data["row"]
        if row:
            self._animating_list.remove(row)
            self._open_list.remove(row)

    def update_cell(self, cell, row):
        cell.part_text_set("label", self.text_func(row))

        if self.bt_left_func:
            cell.part_text_set("edit_button_label", self.bt_left_func(row))

        if self.bt_right_func:
            cell.part_text_set("delete_button_label", self.bt_right_func(row))

        # XXX: we are leaking rows inside animating_list when
        # the user moves and the animation suddenly stops.
        if row in self._animating_list:
            pass
        elif row in self._open_list:
            cell.signal_emit("options,open", "")
        else:
            cell.signal_emit("options,close", "")

    def show_cell(self, cell, x, y, w, h):
        cell.geometry = x, y, w, h
        cell.show()

    def press(self, cell, part, row):
        cell.signal_emit("%s,press" % part, "")

    def release(self, cell, part, row):
        cell.signal_emit("%s,release" % part, "")

    def click(self, cell, part, row):
        if part == "event_area":
            if self.item_click:
                self.item_click(row, self.list, *self.args, **self.kargs)
        elif part == "expand_event_area":
            if not row in self._animating_list:
                self._animating_list.append(row)
            cell.signal_emit("options,open,animating", "")
        elif part == "collapse_event_area":
            if not row in self._animating_list:
                self._animating_list.append(row)
            cell.signal_emit("options,close,animating", "")
        elif part == "edit_button":
            if self.edit_click:
                self.edit_click(row, self.list, *self.args, **self.kargs)
        elif part == "delete_button":
            if self.delete_click:
                self.delete_click(row, self.list, *self.args, **self.kargs)


class FeedItemRenderer(ExpandButtonRenderer):
    terra_type = "Renderer/EtkList/FeedItem"

    def __init__(self, text_func=None, item_click=None, edit_click=None,
                 delete_click=None, bt_left_func=None, bt_right_func=None, *a, **ka):
        ExpandButtonRenderer.__init__(self, text_func, item_click, edit_click,
                                      delete_click, bt_left_func, bt_right_func, *a, **ka)


class OnOffItemRenderer(etk.EdjeKineticRenderer, TerraObject):
    terra_type = "Renderer/EtkList/OnOffItem"

    def __init__(self, ui_func=None, item_click=None):
        if not callable(ui_func):
            raise TypeError("ui_func must be callable")

        etk.EdjeKineticRenderer.__init__(self)
        self.ui_func = ui_func
        self.item_click = item_click

    def create_cell(self, canvas):
        cell = edje.Edje(canvas)
        etk.theme_edje_object_set(cell, self.list.theme_file,
                                  "onoff_item", "canola/conf")
        return cell

    def update_cell(self, cell, row):
        ui_data = self.ui_func(row)
        if ui_data is None:
            return

        text, checked = ui_data
        cell.part_text_set("label", text)
        if checked:
            cell.signal_emit("cell,state,on", "")
        else:
            cell.signal_emit("cell,state,off", "")

    def show_cell(self, cell, x, y, w, h):
        cell.geometry = x, y, w, h
        cell.show()

    def click(self, cell, part, row):
        if self.item_click:
            self.item_click(row, self.list)


class MultipleItemRenderer(etk.EdjeKineticRenderer, TerraObject):
    terra_type = "Renderer/EtkList/MultipleItem"

    def __init__(self):
        etk.EdjeKineticRenderer.__init__(self)

    def render(self, cell, row, list, x, y, w, h):
        return row.renderer.render(cell, row, list, x, y, w, h)


class CheckItemRenderer(etk.EdjeKineticRenderer, TerraObject):
    terra_type = "Renderer/EtkList/CheckItem"

    def __init__(self, ui_func=None, item_click=None):
        if not callable(ui_func):
            raise TypeError("ui_func must be callable")

        etk.EdjeKineticRenderer.__init__(self)
        self.ui_func = ui_func
        self.item_click = item_click

    def create_cell(self, canvas):
        cell = edje.Edje(canvas)
        etk.theme_edje_object_set(cell, self.list.theme_file,
                                  "check_item", "canola/conf")
        return cell

    def update_cell(self, cell, row):
        ui_data = self.ui_func(row)
        if ui_data is None:
            return

        text, checked = ui_data
        cell.part_text_set("label", text)
        if checked:
            cell.signal_emit("cell,state,checked", "")
        else:
            cell.signal_emit("cell,state,unchecked", "")

    def show_cell(self, cell, x, y, w, h):
        cell.geometry = x, y, w, h
        cell.show()

    def click(self, cell, part, row):
        if self.item_click:
            self.item_click(row, self.list)


class SliderRenderer(etk.EdjeKineticRenderer, TerraObject):
    terra_type = "Renderer/EtkList/Slider"

    def __init__(self, text_func=None, item_click=None,
                 get_value=None, get_min_value=None, get_max_value=None,
                 cb_value_change=None, *a, **ka):
        if text_func is None:
            raise ValueError("need a text func")

        etk.EdjeKineticRenderer.__init__(self)
        self.get_value = get_value
        self.get_min_value = get_min_value
        self.get_max_value = get_max_value
        self.text_func = text_func
        self.item_click = item_click
        self.cb_value_change = cb_value_change

        self.args = a
        self.kargs = ka
        self._open_list = []
        self._animating_list = []

    def create_cell(self, canvas):
        cell = edje.Edje(canvas)
        etk.theme_edje_object_set(cell, self.list.theme_file,
                                  "slider", "canola/conf")
        cell.signal_callback_add("options,open,finished", "*",
                                 self._end_open_animation)
        cell.signal_callback_add("options,close,finished", "*",
                                 self._end_close_animation)
        cell.signal_callback_add("drag", "knob",
                                 self._drag_value_set)

        return cell

    def value_set(self, cell, row):
        value = float(self.get_value(row))
        vmin = float(self.get_min_value(row))
        vmax = float(self.get_max_value(row))

        percent = 1 - (vmax - value) / (vmax - vmin)

        cell.part_drag_value_set("knob", percent, 1)
        cell.part_text_set("slide_label", "%.1f" % value)

    def _drag_value_set(self, cell, signal, src):
        row = cell.data["row"]
        vmin = self.get_min_value(row)
        vmax = self.get_max_value(row)

        percent, factor = cell.part_drag_value_get("knob")
        value = vmax - (vmax - vmin) * (1 - percent)
        cell.part_text_set("slide_label", "%.1f" % value)

        if self.cb_value_change:
            self.cb_value_change(row, value)

    def _end_open_animation(self, cell, signal, src):
        row = cell.data["row"]
        if row:
            self._animating_list.remove(row)
            self._open_list.append(row)

    def _end_close_animation(self, cell, signal, src):
        row = cell.data["row"]
        if row:
            self._animating_list.remove(row)
            self._open_list.remove(row)

    def update_cell(self, cell, row):
        cell.part_text_set("label", self.text_func(row))

        self.value_set(cell, row)

        # XXX: we are leaking rows inside animating_list when
        # the user moves and the animation suddenly stops.
        if row in self._animating_list:
            pass
        elif row in self._open_list:
            cell.signal_emit("options,open", "")
        else:
            cell.signal_emit("options,close", "")

    def show_cell(self, cell, x, y, w, h):
        cell.geometry = x, y, w, h
        cell.show()
        self.value_set(cell, cell.data["row"])

    def press(self, cell, part, row):
        cell.signal_emit("%s,press" % part, "")

    def release(self, cell, part, row):
        cell.signal_emit("%s,release" % part, "")

    def click(self, cell, part, row):
        if part == "event_area":
            if self.item_click:
                self.item_click(row, self.list, *self.args, **self.kargs)
        elif part == "expand_event_area":
            if not row in self._animating_list:
                self._animating_list.append(row)
            cell.signal_emit("options,open,animating", "")
        elif part == "collapse_event_area":
            if not row in self._animating_list:
                self._animating_list.append(row)
            cell.signal_emit("options,close,animating", "")
