# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.


import pygtk
pygtk.require('2.0')

import desktopcouch.tests as test_environment

from desktopcouch.pair.couchdb_pairing import couchdb_io
from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import Record
from twisted.trial import unittest
import uuid
import os
import httplib2
import socket
URI = None  # use autodiscovery that desktopcouch.tests permits.

class TestCouchdbIo(unittest.TestCase):

    def setUp(self):
        """setup each test"""
        self.mgt_database = CouchDatabase('management', create=True, uri=URI,
                ctx=test_environment.test_context)
        self.foo_database = CouchDatabase('foo', create=True, uri=URI,
                ctx=test_environment.test_context)
        #create some records to pull out and test
        self.foo_database.put_record(Record({
            "key1_1": "val1_1", "key1_2": "val1_2", "key1_3": "val1_3",
            "record_type": "test.com"}))
        self.foo_database.put_record(Record({
            "key2_1": "val2_1", "key2_2": "val2_2", "key2_3": "val2_3",
            "record_type": "test.com"}))
        self.foo_database.put_record(Record({
            "key13_1": "va31_1", "key3_2": "val3_2", "key3_3": "val3_3",
            "record_type": "test.com"}))

    def tearDown(self):
        """tear down each test"""
        del self.mgt_database._server['management']
        del self.mgt_database._server['foo']


    def test_obsfuscation(self):
        t = {'url': 'https://couchdb.one.ubuntu.com/u%2Fb2%2Fc8%2F276%2Ftest', 'auth': {'oauth': {'consumer_secret': 'SeCrEtSe', 'token': '3XRjQrWX92TTTJFDTWJJ', 'consumer_key': 'ubuntuone', 'token_secret': 'jBmSeCrEtawkefwklefliwuregqwlkeh347wq87w4fiuq4fyu3q4fiqwu4fqwfiqufM6xjsPwSeCrEt4'}}}
        cleaned_t = couchdb_io.obsfuscate(t)
        self.failIf("SeCrEt" in str(cleaned_t), {'url': 'https://couchdb.one.ubuntu.com/u%2Fb2%2Fc8%2F276%2Ftest', 'auth': {'oauth': {'consumer_secret': 'HiddenHidd', 'token': '3XRjQrWX92TTTJFDTWJJ', 'consumer_key': 'ubuntuone', 'token_secret': 'HiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHi'}}})

        self.assertEqual(couchdb_io.obsfuscate(""), "")
        self.assertEqual(couchdb_io.obsfuscate({}), {})
        self.assertEqual(couchdb_io.obsfuscate({1:{}}), {1:{}})
        self.assertEqual(couchdb_io.obsfuscate({1:1}), {1:1})


    def test_put_static_paired_service(self):
        service_name = "dummyfortest"
        oauth_data = {
                "consumer_key": str("abcdef"),
                "consumer_secret": str("ghighjklm"),
                "token": str("opqrst"),
                "token_secret": str("uvwxyz"),
            }
        couchdb_io.put_static_paired_service(oauth_data, service_name, uri=URI, ctx=test_environment.test_context)
        pairings = list(couchdb_io.get_pairings(ctx=test_environment.test_context))

    def test_put_dynamic_paired_host(self):
        hostname = "host%d" % (os.getpid(),)
        remote_uuid = str(uuid.uuid4())
        oauth_data = {
                "consumer_key": str("abcdef"),
                "consumer_secret": str("ghighjklm"),
                "token": str("opqrst"),
                "token_secret": str("uvwxyz"),
            }

        couchdb_io.put_dynamic_paired_host(hostname, remote_uuid, oauth_data,
                uri=URI, ctx=test_environment.test_context)
        couchdb_io.put_dynamic_paired_host(hostname, remote_uuid, oauth_data,
                uri=URI, ctx=test_environment.test_context)
        couchdb_io.put_dynamic_paired_host(hostname, remote_uuid, oauth_data,
                uri=URI, ctx=test_environment.test_context)

        pairings = list(couchdb_io.get_pairings(ctx=test_environment.test_context))
        self.assertEqual(3, len(pairings))
        self.assertEqual(pairings[0].value["oauth"], oauth_data)
        self.assertEqual(pairings[0].value["server"], hostname)
        self.assertEqual(pairings[0].value["pairing_identifier"], remote_uuid)

        for i, row in enumerate(pairings):
            couchdb_io.remove_pairing(row.id, i == 1, ctx=test_environment.test_context)

        pairings = list(couchdb_io.get_pairings(ctx=test_environment.test_context))
        self.assertEqual(0, len(pairings))


    def test_get_database_names_replicatable_bad_server(self):
        hostname = "test.desktopcouch.example.com"
        try:
            socket.gethostbyname(hostname)
            raise unittest.SkipTest("nxdomain hijacked")
        except socket.gaierror:
            pass

        try:
            names = couchdb_io.get_database_names_replicatable(
                uri='http://' + hostname + ':9/')
            self.assertEqual(set(), names)
        except httplib2.ServerNotFoundError:
            pass

    def test_get_database_names_replicatable(self):
        names = couchdb_io.get_database_names_replicatable(uri=URI, ctx=test_environment.test_context)
        self.assertFalse('management' in names)
        self.assertTrue('foo' in names)

    def test_get_my_host_unique_id(self):
        got = couchdb_io.get_my_host_unique_id(uri=URI, ctx=test_environment.test_context)
        again = couchdb_io.get_my_host_unique_id(uri=URI, ctx=test_environment.test_context)
        self.assertEquals(len(got), 1)
        self.assertEquals(got, again)

    def test_mkuri(self):
        uri = couchdb_io.mkuri(
            'fnord.org', 55241, has_ssl=True, path='a/b/c',
            auth_pair=('f o o', 'b=a=r'))
        self.assertEquals(
            'https://f%20o%20o:b%3Da%3Dr@fnord.org:55241/a/b/c', uri)

    def Xtest_replication_good(self):
        pass

    def Xtest_replication_no_oauth_remote(self):
        pass

    def Xtest_replication_bad_oauth_remote(self):
        pass

    def Xtest_replication_no_oauth_local(self):
        pass

    def Xtest_replication_bad_oauth_local(self):
        pass


if __name__ == "__main__":
    unittest.main()
