/*
 * Copyright (C) 2010 Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_buffer.c
 *
 * Description:    Implementation of Multimedia Buffer management for Gstreamer
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */


/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <errno.h>
#include <gst/gst.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>

#include "mfw_gst_buffer.h"

#ifdef USE_VPU_MEM
#include "vpu_io.h"
#else
#include "linux/ipu.h"
#endif


GST_DEBUG_CATEGORY (mfw_gst_buffer_debug);
#define GST_CAT_DEFAULT mfw_gst_buffer_debug


#define DEBUG_ERROR(format,...)    printf(format, ##__VA_ARGS__)
#define DEBUG_FLOW(format,...)     printf(format, ##__VA_ARGS__)
#define DEBUG_MESSAGE(format,...)  printf(format, ##__VA_ARGS__)


#ifdef USE_VPU_MEM
typedef vpu_mem_desc mem_desc;
#define PHY_MEM_ALLOC_IOCTL_ID VPU_IOC_PHYMEM_FREE
#define PHY_MEM_FREE_IOCTL_ID VPU_IOC_PHYMEM_ALLOC
#define MEMORY_DEVICE_NAME "/dev/mxc_vpu"
#define DESC2PHYADDRESS(memdesc) ((memdesc)->phy_addr)
#define DESC2SIZE(memdesc) ((memdesc)->size)
#else
typedef ipu_mem_info mem_desc;
#define PHY_MEM_ALLOC_IOCTL_ID IPU_ALOC_MEM
#define PHY_MEM_FREE_IOCTL_ID IPU_FREE_MEM
#define MEMORY_DEVICE_NAME "/dev/mxc_ipu"
#define DESC2PHYADDRESS(memdesc) ((memdesc)->paddr)
#define DESC2SIZE(memdesc) ((memdesc)->size)
#endif

typedef struct _HWBufDesc{
    unsigned long phyaddr;
    unsigned long virtaddr;
    mem_desc memdesc;
    int size;
    void * priv;
    struct _HWBufDesc * prev;
    struct _HWBufDesc * next;
    struct _HWBufDesc * link;
}HWBufDesc;



typedef struct _HWBufAllocator{
    GMutex *    lock;
    int mem_fd;
    int cnt;
    int freecnt;
    HWBufDesc * free;
    HWBufDesc * all;
    int descontruct;
    struct _HWBufAllocator * next;
}HWBufAllocator;



#define FSL_MM_IOCTL(device, request, errorroute, ...)\
    do{\
        int ret;\
        if ((ret = ioctl((device), (request), ##__VA_ARGS__))<0){\
            GST_ERROR("%s:%d ioctl aaa error, return %d\n", __FILE__, __LINE__, ret);\
            goto errorroute;\
        }\
    }while(0)

/* For deconstructor function */
static HWBufAllocator * glocal_allocator = NULL;

/*=============================================================================
FUNCTION:           destory_hw_buf_allocator_lock

DESCRIPTION:        This funtion free the hardware buffer resource in safe mode.
                    

ARGUMENTS PASSED:   None

RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void destory_hw_buf_allocator_lock(void * allochandle, int force)
{
    HWBufAllocator * allocator = (HWBufAllocator *)allochandle;
    HWBufDesc * tmp = allocator->all;
    HWBufDesc * tmpnext;

    /* 
     * did not free any software resource when called by deconstructor, 
     * due to other element may access it, and it will finally free 
     *  by loader 
     */

    if (force==0) {
        HWBufAllocator * al1 = glocal_allocator;
        if (al1==allocator){
            glocal_allocator = allocator->next;
        }else{
            while(al1->next!=allocator)
                al1=al1->next;
            al1->next = allocator->next;
        }
    }
    
    while(tmp){
        allocator->cnt--;
        tmpnext = tmp->link;
        munmap(tmp->virtaddr, tmp->size);
        mem_desc * memdesc = &tmp->memdesc;
        FSL_MM_IOCTL(allocator->mem_fd, PHY_MEM_FREE_IOCTL_ID, error, memdesc);
        if (force==0)
            g_free(tmp);
        tmp = tmpnext;
    }
    allocator->free=NULL;
    allocator->all=NULL;

    g_mutex_unlock(allocator->lock);
    
    if (force==0){
        g_mutex_free(allocator->lock);
        g_free(allocator);
    }
    
    GST_INFO("Hwbuf Allocator destroied, force=%d!\n", force);
error:
    return;
}





/*=============================================================================
FUNCTION:           destory_hw_buf_allocator

DESCRIPTION:        This funtion free the hardware buffer resource.
                    

ARGUMENTS PASSED:   None

RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

void destory_hw_buf_allocator(void * allochandle)
{
    HWBufAllocator * allocator = (HWBufAllocator *)allochandle;
    g_mutex_lock(allocator->lock);
    destory_hw_buf_allocator_lock(allochandle, 0);
}


/*=============================================================================
FUNCTION:           setHWBufferAllocatorSelfDescontruct

DESCRIPTION:        This funtion set the descontructor flag.
                    

ARGUMENTS PASSED:   None

RETURN VALUE:       The buffer allocator pointer.

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
void setHWBufferAllocatorSelfDescontruct(void * allochandle)
    
{
    HWBufAllocator * allocator = (HWBufAllocator *)allochandle;
    allocator->descontruct=1;
}

/*=============================================================================
FUNCTION:           mfw_new_hw_buffer

DESCRIPTION:        This funtion request a physical memory buffer.
                    

ARGUMENTS PASSED:   None

RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

HWBufDesc * mfw_new_hw_buffer(void * ahandle, int size)
{
    HWBufAllocator * allocator = (HWBufAllocator *)ahandle;
    HWBufDesc * bufdesc;

    if (ahandle==NULL)
        return NULL;


    GST_DEBUG("Request buffer size:%d for handle %p\n",size, ahandle);
    
    g_mutex_lock(allocator->lock);
    
    bufdesc = allocator->free;

    while(bufdesc){
        if (bufdesc->size>=size)
            break;
        bufdesc=bufdesc->next;
    }
    if (bufdesc) {
        if (bufdesc->prev){
            bufdesc->prev->next = bufdesc->next;
        }else{
            allocator->free = bufdesc->next;
        }
        if (bufdesc->next){
            bufdesc->next->prev = bufdesc->prev;
        }
        allocator->freecnt--;
    }

    else {
        
        bufdesc = g_malloc(sizeof(HWBufDesc));
        if (bufdesc==NULL){
            goto error;
        }
        mem_desc * memdesc =  &bufdesc->memdesc;
        memset(memdesc, 0, sizeof(mem_desc));
        DESC2SIZE(memdesc) = size;
        FSL_MM_IOCTL(allocator->mem_fd, PHY_MEM_ALLOC_IOCTL_ID, error, memdesc);
        bufdesc->phyaddr = DESC2PHYADDRESS(memdesc);
        bufdesc->virtaddr =  mmap(NULL, size,
				    PROT_READ | PROT_WRITE, MAP_SHARED,
				    allocator->mem_fd, bufdesc->phyaddr);
        if ((int)bufdesc->virtaddr==-1){
            FSL_MM_IOCTL(allocator->mem_fd, PHY_MEM_FREE_IOCTL_ID, error, memdesc);
            goto error;
        }
        bufdesc->size = size;
        bufdesc->priv = allocator;
        bufdesc->link = allocator->all;
        allocator->all = bufdesc;
        allocator->cnt++;
        
    }
    g_mutex_unlock(allocator->lock);
    return bufdesc;
    
error:
    g_print("Request buffer failed, total buffer count:%d, fee buffer count:%d\n", 
        allocator->cnt,allocator->freecnt);

    if (bufdesc){
        g_free(bufdesc);
    }
    g_mutex_unlock(allocator->lock);
    
    return NULL;
}

/*=============================================================================
FUNCTION:           mfw_free_hw_buffer

DESCRIPTION:        This funtion free a physical memory buffer.
                    

ARGUMENTS PASSED:   None

RETURN VALUE:       

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

void mfw_free_hw_buffer( HWBufDesc * bufdesc)
{
    HWBufAllocator * allocator = (HWBufAllocator *)bufdesc->priv;

    g_mutex_lock(allocator->lock);
    bufdesc->prev = NULL;
    bufdesc->next = allocator->free;
    if (allocator->free){
        allocator->free->prev=bufdesc;
    }
    allocator->free = bufdesc;
    allocator->freecnt++;
  

    if ((allocator->freecnt==allocator->cnt)&&(allocator->descontruct)){
        destory_hw_buf_allocator_lock(allocator, 0);
        return;
    }
    
    g_mutex_unlock(allocator->lock);
}



/*=============================================================================
FUNCTION:           mfw_create_hw_buf_allocator

DESCRIPTION:        This funtion registers the buffer allocator structure.
                    

ARGUMENTS PASSED:   None

RETURN VALUE:       The buffer allocator pointer.

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
void *mfw_create_hw_buf_allocator()
{
    HWBufAllocator * allocator = NULL;
    allocator = g_malloc(sizeof(HWBufAllocator));
    if (allocator==NULL){
        g_print("can not allocate memory for hwbuf allocator\n");
        goto error;
    }

    memset(allocator, 0, sizeof(HWBufAllocator));
    allocator->lock = g_mutex_new();

    
    if ((allocator->mem_fd=open(MEMORY_DEVICE_NAME, O_RDWR))<0){
        g_print("can not open %s device\n", MEMORY_DEVICE_NAME);
    }

    allocator->next = glocal_allocator;
    glocal_allocator = allocator;

    return allocator;
    
error:
    if (allocator)
        g_free(allocator);
    
    return NULL;
}

/*=============================================================================
FUNCTION:           mfw_gst_buffer_finalize

DESCRIPTION:        This funtion free the buffer resource.

ARGUMENTS PASSED:
        MFWGstBuffer        -  The buffer structure pointer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_buffer_finalize(MFWGstBuffer *buffer)
{
    if (buffer->priv){
        mfw_free_hw_buffer(buffer->priv);
        buffer->priv=NULL;
    }else{
        if (GST_BUFFER_DATA(GST_BUFFER_CAST(buffer)))
            g_free(GST_BUFFER_DATA(GST_BUFFER_CAST(buffer)));
    }


    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_buffer_new

DESCRIPTION:        This funtion free the buffer resource.

ARGUMENTS PASSED:
        GstElement        -  The Gstreamer Element pointer.
        size              -  Request buffer size.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

MFWGstBuffer *mfw_gst_buffer_new(GstElement * obj, int size)
{
    MFWGstBuffer * buf=NULL;
    char * data;

    buf = (MFWGstBuffer *)gst_mini_object_new(MFW_GST_TYPE_BUFFER);
    
    if (obj->object._gst_reserved == NULL){
        obj->object._gst_reserved = (gpointer) mfw_create_hw_buf_allocator();
    }
    buf->priv = mfw_new_hw_buffer(obj->object._gst_reserved,size);
    if (buf->priv == NULL){
        (GST_BUFFER_DATA(GST_BUFFER_CAST(buf))) = g_malloc(size);
        if ((GST_BUFFER_DATA(GST_BUFFER_CAST(buf)))==NULL){
            goto error;
        }
        GST_BUFFER_FLAG_UNSET(&buf->buffer, GST_BUFFER_FLAG_LAST);
    }else{
        GST_BUFFER_DATA(GST_BUFFER_CAST(buf)) = ((HWBufDesc *)buf->priv)->virtaddr;
        GST_BUFFER_OFFSET(GST_BUFFER_CAST(buf)) = ((HWBufDesc *)buf->priv)->phyaddr;
        GST_BUFFER_FLAG_SET(&buf->buffer, GST_BUFFER_FLAG_LAST);
    }
    GST_BUFFER_SIZE(buf)=size;
    return buf;
error:
    if (buf)
        gst_buffer_unref(buf);
    return NULL;
}

/*=============================================================================
FUNCTION:           mfw_gst_buffer_class_init

DESCRIPTION:        This funtion registers the  funtions used by the
                    buffer class for common use.

ARGUMENTS PASSED:
        g_class        -   class from which the mini objext is derived
        class_data     -   global class data

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static void
mfw_gst_buffer_class_init(gpointer g_class, gpointer class_data)
{
    GstMiniObjectClass *mini_object_class = GST_MINI_OBJECT_CLASS(g_class);
    mini_object_class->finalize = (GstMiniObjectFinalizeFunction)
        mfw_gst_buffer_finalize;
    return;

}


/*=============================================================================
FUNCTION:           mfw_gst_buffer_get_type

DESCRIPTION:        This funtion registers the  buffer class
                    for common use.

ARGUMENTS PASSED:   None

RETURN VALUE:       return the registered buffer class

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

GType mfw_gst_buffer_get_type(void)
{

    static GType _mfw_gst_buffer_type;

    if (G_UNLIKELY(_mfw_gst_buffer_type == 0)) {
    static const GTypeInfo mfw_buf_info = {
        sizeof(GstBufferClass),
        NULL,
        NULL,
        mfw_gst_buffer_class_init,
        NULL,
        NULL,
        sizeof(MFWGstBuffer),
        0,
        NULL,
        NULL
    };
    _mfw_gst_buffer_type =
        g_type_register_static(GST_TYPE_BUFFER, "MFWGstBuffer",
                   &mfw_buf_info, 0);
    }

    GST_DEBUG_CATEGORY_INIT(mfw_gst_buffer_debug,
			    "mfw_gstbuffer", 0,
			    "FreeScale's Common Buffer");

    return _mfw_gst_buffer_type;
}


/*=============================================================================
FUNCTION:           mfw_gst_buf_dest

DESCRIPTION:        This funtion free the hardware buffer resource in case 
                       of exception.

ARGUMENTS PASSED:   None

RETURN VALUE:       None

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
void __attribute__ ((destructor)) mfw_gst_buf_dest(void);

void mfw_gst_buf_dest(void)
{
    HWBufAllocator * hd = glocal_allocator; 
    HWBufAllocator * hdnext;
    while(hd){
        hdnext = hd->next;
        g_mutex_lock(hd->lock);
        destory_hw_buf_allocator_lock(hd, 1);
        hd = hdnext;
    }
    
}



