/*!
 * @section LICENSE
 * (C) Copyright 2013 Bosch Sensortec GmbH All Rights Reserved
 *
 * This software program is licensed subject to the GNU General
 * Public License (GPL).Version 2,June 1991,
 * available at http://www.fsf.org/copyleft/gpl.html
 *
 * @filename bmg160.c
 * @date    2013/11/25
 * @id       "079d340"
 * @version  1.5
 *
 * @brief    BMG160API
*/


#ifdef __KERNEL__
#include <linux/kernel.h>
#include <linux/unistd.h>
#include <linux/types.h>
#include <linux/string.h>
#else
#include <unistd.h>
#include <sys/types.h>
#include <string.h>
#endif

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/regulator/consumer.h>


#define BMG_USE_BASIC_I2C_FUNC 1

#ifdef CONFIG_VLV2_PLAT_CLK
#include <linux/vlv2_plat_clock.h>
#endif

#include <linux/atomisp_gmin_platform.h>
#include <linux/acpi.h>
#include <linux/io.h>

enum { V2P8_GPIO_UNSET = -2, V2P8_GPIO_NONE = -1 };
static int v2p8_gpio = V2P8_GPIO_UNSET;

/*
 * Something of a hack. The CHT RVP board drives camera 1.8v from an
 * external regulator instead of the PMIC just like ECS E7 board, see the
 * comments above.
 */
enum { V1P8_GPIO_UNSET = -2, V1P8_GPIO_NONE = -1 };
static int v1p8_gpio = V1P8_GPIO_UNSET;






/* sensor specific */
#define SENSOR_NAME "bmg160"

#define SENSOR_CHIP_ID_BMG (0x0f)
#define CHECK_CHIP_ID_TIME_MAX   5

#define BMG_REG_NAME(name) BMG160_##name
#define BMG_VAL_NAME(name) BMG160_##name
#define BMG_CALL_API(name) bmg160_##name

#define BMG_I2C_WRITE_DELAY_TIME 1

/* generic */
#define BMG_MAX_RETRY_I2C_XFER (100)
#define BMG_MAX_RETRY_WAKEUP (5)
#define BMG_MAX_RETRY_WAIT_DRDY (100)

#define BMG_DELAY_MIN (1)
#define BMG_DELAY_DEFAULT (200)

#define BMG_VALUE_MAX (32767)
#define BMG_VALUE_MIN (-32768)

#define BYTES_PER_LINE (16)

#define BMG_SELF_TEST 0

#define BMG_SOFT_RESET_VALUE                0xB6


#define MAX_FIFO_F_LEVEL 100
#define MAX_FIFO_F_BYTES 8
#ifdef __KERNEL__
#define BMG160_U16 unsigned short       /* 16 bit achieved with short */
#define BMG160_S16 signed short
#define BMG160_S32 signed int           /* 32 bit achieved with int   */
#else
#include <limits.h> /*needed to test integer limits */


/* find correct data type for signed/unsigned 16 bit variables \
by checking max of unsigned variant */
#if USHRT_MAX == 0xFFFF
		/* 16 bit achieved with short */
		#define BMG160_U16 unsigned short
		#define BMG160_S16 signed short
#elif UINT_MAX == 0xFFFF
		/* 16 bit achieved with int */
		#define BMG160_U16 unsigned int
		#define BMG160_S16 signed int
#else
		#error BMG160_U16 and BMG160_S16 could not be
		#error defined automatically, please do so manually
#endif

/* find correct data type for signed 32 bit variables */
#if INT_MAX == 0x7FFFFFFF
		/* 32 bit achieved with int */
		#define BMG160_S32 signed int
#elif LONG_MAX == 0x7FFFFFFF
		/* 32 bit achieved with long int */
		#define BMG160_S32 signed long int
#else
		#error BMG160_S32 could not be
		#error defined automatically, please do so manually
#endif
#endif

/**\brief defines the calling parameter types of the BMG160_WR_FUNCTION */
#define BMG160_BUS_WR_RETURN_TYPE char

/**\brief links the order of parameters defined in
BMG160_BUS_WR_PARAM_TYPE to function calls used inside the API*/
#define BMG160_BUS_WR_PARAM_TYPES unsigned char,\
unsigned char *, unsigned char

/**\brief links the order of parameters defined in
BMG160_BUS_WR_PARAM_TYPE to function calls used inside the API*/
#define BMG160_BUS_WR_PARAM_ORDER(register_addr,\
register_data, wr_len)

/* never change this line */
#define BMG160_BUS_WRITE_FUNC(register_addr,\
register_data, wr_len) bus_write(register_addr,\
register_data, wr_len)
/**\brief defines the return parameter type of the BMG160_RD_FUNCTION
*/
#define BMG160_BUS_RD_RETURN_TYPE char
/**\brief defines the calling parameter types of the BMG160_RD_FUNCTION
*/
#define BMG160_BUS_RD_PARAM_TYPES unsigned char,\
unsigned char *, unsigned char
/**\brief links the order of parameters defined in \
BMG160_BUS_RD_PARAM_TYPE to function calls used inside the API
*/
#define BMG160_BUS_RD_PARAM_ORDER (register_addr,\
register_data, rd_len)
/* never change this line */
#define BMG160_BUS_READ_FUNC(register_addr,\
register_data, rd_len)bus_read(register_addr,\
register_data, rd_len)
/**\brief defines the return parameter type of the BMG160_RD_FUNCTION
*/
#define BMG160_BURST_RD_RETURN_TYPE char
/**\brief defines the calling parameter types of the BMG160_RD_FUNCTION
*/
#define BMG160_BURST_RD_PARAM_TYPES \
unsigned char, unsigned char *, signed int
/**\brief links the order of parameters defined in \
BMG160_BURST_RD_PARAM_TYPE to function calls used inside the API
*/
#define BMG160_BURST_RD_PARAM_ORDER (register_addr,\
register_data , rd_len)
/* never change this line */
#define BMG160_BURST_READ_FUNC(register_addr,\
register_data, rd_len)burst_read(\
register_addr, register_data, rd_len)
/**\brief defines the return parameter type of the BMG160_DELAY_FUNCTION
*/
#define BMG160_DELAY_RETURN_TYPE void
/* never change this line */
#define BMG160_DELAY_FUNC(delay_in_msec)\
		delay_func(delay_in_msec)
#define BMG160_RETURN_FUNCTION_TYPE			int
/**< This refers BMG160 return type as char */


/*Define of registers*/

/* Hard Wired */
#define BMG160_CHIP_ID_ADDR						0x00
/**<Address of Chip ID Register*/


/* Data Register */
#define BMG160_RATE_X_LSB_ADDR                   0x02
/**<        Address of X axis Rate LSB Register       */
#define BMG160_RATE_X_MSB_ADDR                   0x03
/**<        Address of X axis Rate MSB Register       */
#define BMG160_RATE_Y_LSB_ADDR                   0x04
/**<        Address of Y axis Rate LSB Register       */
#define BMG160_RATE_Y_MSB_ADDR                   0x05
/**<        Address of Y axis Rate MSB Register       */
#define BMG160_RATE_Z_LSB_ADDR                   0x06
/**<        Address of Z axis Rate LSB Register       */
#define BMG160_RATE_Z_MSB_ADDR                   0x07
/**<        Address of Z axis Rate MSB Register       */
#define BMG160_TEMP_ADDR                        0x08
/**<        Address of Temperature Data LSB Register  */

/* Status Register */
#define BMG160_INT_STATUS0_ADDR                 0x09
/**<        Address of Interrupt status Register 0    */
#define BMG160_INT_STATUS1_ADDR                 0x0A
/**<        Address of Interrupt status Register 1    */
#define BMG160_INT_STATUS2_ADDR                 0x0B
/**<        Address of Interrupt status Register 2    */
#define BMG160_INT_STATUS3_ADDR                 0x0C
/**<        Address of Interrupt status Register 3    */
#define BMG160_FIFO_STATUS_ADDR                 0x0E
/**<        Address of FIFO status Register           */

/* Control Register */
#define BMG160_RANGE_ADDR                  0x0F
/**<        Address of Range address Register     */
#define BMG160_BW_ADDR                     0x10
/**<        Address of Bandwidth Register         */
#define BMG160_MODE_LPM1_ADDR              0x11
/**<        Address of Mode LPM1 Register         */
#define BMG160_MODE_LPM2_ADDR              0x12
/**<        Address of Mode LPM2 Register         */
#define BMG160_RATED_HBW_ADDR              0x13
/**<        Address of Rate HBW Register          */
#define BMG160_BGW_SOFTRESET_ADDR          0x14
/**<        Address of BGW Softreset Register      */
#define BMG160_INT_ENABLE0_ADDR            0x15
/**<        Address of Interrupt Enable 0             */
#define BMG160_INT_ENABLE1_ADDR            0x16
/**<        Address of Interrupt Enable 1             */
#define BMG160_INT_MAP_0_ADDR              0x17
/**<        Address of Interrupt MAP 0                */
#define BMG160_INT_MAP_1_ADDR              0x18
/**<        Address of Interrupt MAP 1                */
#define BMG160_INT_MAP_2_ADDR              0x19
/**<        Address of Interrupt MAP 2                */
#define BMG160_INT_0_ADDR                  0x1A
/**<        Address of Interrupt 0 register   */
#define BMG160_INT_1_ADDR                  0x1B
/**<        Address of Interrupt 1 register   */
#define BMG160_INT_2_ADDR                  0x1C
/**<        Address of Interrupt 2 register   */
#define BMG160_INT_4_ADDR                  0x1E
/**<        Address of Interrupt 4 register   */
#define BMG160_RST_LATCH_ADDR              0x21
/**<        Address of Reset Latch Register           */
#define BMG160_HIGH_TH_X_ADDR              0x22
/**<        Address of High Th x Address register     */
#define BMG160_HIGH_DUR_X_ADDR             0x23
/**<        Address of High Dur x Address register    */
#define BMG160_HIGH_TH_Y_ADDR              0x24
/**<        Address of High Th y  Address register    */
#define BMG160_HIGH_DUR_Y_ADDR             0x25
/**<        Address of High Dur y Address register    */
#define BMG160_HIGH_TH_Z_ADDR              0x26
/**<        Address of High Th z Address register  */
#define BMG160_HIGH_DUR_Z_ADDR             0x27
/**<        Address of High Dur z Address register  */
#define BMG160_SOC_ADDR                        0x31
/**<        Address of SOC register        */
#define BMG160_A_FOC_ADDR                      0x32
/**<        Address of A_FOC Register        */
#define BMG160_TRIM_NVM_CTRL_ADDR          0x33
/**<        Address of Trim NVM control register      */
#define BMG160_BGW_SPI3_WDT_ADDR           0x34
/**<        Address of BGW SPI3,WDT Register           */


/* Trim Register */
#define BMG160_OFC1_ADDR                   0x36
/**<        Address of OFC1 Register          */
#define BMG160_OFC2_ADDR                       0x37
/**<        Address of OFC2 Register          */
#define BMG160_OFC3_ADDR                   0x38
/**<        Address of OFC3 Register          */
#define BMG160_OFC4_ADDR                   0x39
/**<        Address of OFC4 Register          */
#define BMG160_TRIM_GP0_ADDR               0x3A
/**<        Address of Trim GP0 Register              */
#define BMG160_TRIM_GP1_ADDR               0x3B
/**<        Address of Trim GP1 Register              */
#define BMG160_SELF_TEST_ADDR              0x3C
/**<        Address of BGW Self test Register           */

/* Control Register */
#define BMG160_FIFO_CGF1_ADDR              0x3D
/**<        Address of FIFO CGF0 Register             */
#define BMG160_FIFO_CGF0_ADDR              0x3E
/**<        Address of FIFO CGF1 Register             */

/* Data Register */
#define BMG160_FIFO_DATA_ADDR              0x3F
/**<        Address of FIFO Data Register             */

/* Rate X LSB Register */
#define BMG160_RATE_X_LSB_VALUEX__POS        0

/**< Last 8 bits of RateX LSB Registers */
#define BMG160_RATE_X_LSB_VALUEX__LEN        8
#define BMG160_RATE_X_LSB_VALUEX__MSK        0xFF
#define BMG160_RATE_X_LSB_VALUEX__REG        BMG160_RATE_X_LSB_ADDR

/* Rate Y LSB Register */
/**<  Last 8 bits of RateY LSB Registers */
#define BMG160_RATE_Y_LSB_VALUEY__POS        0
#define BMG160_RATE_Y_LSB_VALUEY__LEN        8
#define BMG160_RATE_Y_LSB_VALUEY__MSK        0xFF
#define BMG160_RATE_Y_LSB_VALUEY__REG        BMG160_RATE_Y_LSB_ADDR

/* Rate Z LSB Register */
/**< Last 8 bits of RateZ LSB Registers */
#define BMG160_RATE_Z_LSB_VALUEZ__POS        0
#define BMG160_RATE_Z_LSB_VALUEZ__LEN        8
#define BMG160_RATE_Z_LSB_VALUEZ__MSK        0xFF
#define BMG160_RATE_Z_LSB_VALUEZ__REG        BMG160_RATE_Z_LSB_ADDR

/* Interrupt status 0 Register */
   /**< 2th bit of Interrupt status 0 register */
#define BMG160_INT_STATUS0_ANY_INT__POS     2
#define BMG160_INT_STATUS0_ANY_INT__LEN     1
#define BMG160_INT_STATUS0_ANY_INT__MSK     0x04
#define BMG160_INT_STATUS0_ANY_INT__REG     BMG160_INT_STATUS0_ADDR

/**< 4th bit of Interrupt status 1 register */
#define BMG160_INT_MAP_1_INT1DATA__POS           0
#define BMG160_INT_MAP_1_INT1DATA__LEN           1
#define BMG160_INT_MAP_1_INT1DATA__MSK           0x1
#define BMG160_INT_MAP_1_INT1DATA__REG           BMG160_INT_MAP_1_ADDR

/**< 1st bit of Interrupt status 0 register */
#define BMG160_INT_STATUS0_HIGH_INT__POS    1
#define BMG160_INT_STATUS0_HIGH_INT__LEN    1
#define BMG160_INT_STATUS0_HIGH_INT__MSK    0x02
#define BMG160_INT_STATUS0_HIGH_INT__REG    BMG160_INT_STATUS0_ADDR

 /**< 1st and 2nd bit of Interrupt status 0 register */
#define BMG160_INT_STATUSZERO__POS    1
#define BMG160_INT_STATUSZERO__LEN    2
#define BMG160_INT_STATUSZERO__MSK    0x06
#define BMG160_INT_STATUSZERO__REG    BMG160_INT_STATUS0_ADDR

/* Interrupt status 1 Register */
/**< 7th bit of Interrupt status 1 register */
#define BMG160_INT_STATUS1_DATA_INT__POS           7
#define BMG160_INT_STATUS1_DATA_INT__LEN           1
#define BMG160_INT_STATUS1_DATA_INT__MSK           0x80
#define BMG160_INT_STATUS1_DATA_INT__REG           BMG160_INT_STATUS1_ADDR

 /**< 6th bit of Interrupt status 1 register */
#define BMG160_INT_STATUS1_AUTO_OFFSET_INT__POS    6
#define BMG160_INT_STATUS1_AUTO_OFFSET_INT__LEN    1
#define BMG160_INT_STATUS1_AUTO_OFFSET_INT__MSK    0x40
#define BMG160_INT_STATUS1_AUTO_OFFSET_INT__REG    BMG160_INT_STATUS1_ADDR

/**< 5th bit of Interrupt status 1 register */
#define BMG160_INT_STATUS1_FAST_OFFSET_INT__POS    5
#define BMG160_INT_STATUS1_FAST_OFFSET_INT__LEN    1
#define BMG160_INT_STATUS1_FAST_OFFSET_INT__MSK    0x20
#define BMG160_INT_STATUS1_FAST_OFFSET_INT__REG    BMG160_INT_STATUS1_ADDR

/**< 4th bit of Interrupt status 1 register */
#define BMG160_INT_STATUS1_FIFO_INT__POS           4
#define BMG160_INT_STATUS1_FIFO_INT__LEN           1
#define BMG160_INT_STATUS1_FIFO_INT__MSK           0x10
#define BMG160_INT_STATUS1_FIFO_INT__REG           BMG160_INT_STATUS1_ADDR

/**< MSB 4 bits of Interrupt status1 register */
#define BMG160_INT_STATUSONE__POS           4
#define BMG160_INT_STATUSONE__LEN           4
#define BMG160_INT_STATUSONE__MSK           0xF0
#define BMG160_INT_STATUSONE__REG           BMG160_INT_STATUS1_ADDR

/* Interrupt status 2 Register */
/**< 3th bit of Interrupt status 2 register */
#define BMG160_INT_STATUS2_ANY_SIGN_INT__POS     3
#define BMG160_INT_STATUS2_ANY_SIGN_INT__LEN     1
#define BMG160_INT_STATUS2_ANY_SIGN_INT__MSK     0x08
#define BMG160_INT_STATUS2_ANY_SIGN_INT__REG     BMG160_INT_STATUS2_ADDR

/**< 2th bit of Interrupt status 2 register */
#define BMG160_INT_STATUS2_ANY_FIRSTZ_INT__POS   2
#define BMG160_INT_STATUS2_ANY_FIRSTZ_INT__LEN   1
#define BMG160_INT_STATUS2_ANY_FIRSTZ_INT__MSK   0x04
#define BMG160_INT_STATUS2_ANY_FIRSTZ_INT__REG   BMG160_INT_STATUS2_ADDR

/**< 1st bit of Interrupt status 2 register */
#define BMG160_INT_STATUS2_ANY_FIRSTY_INT__POS   1
#define BMG160_INT_STATUS2_ANY_FIRSTY_INT__LEN   1
#define BMG160_INT_STATUS2_ANY_FIRSTY_INT__MSK   0x02
#define BMG160_INT_STATUS2_ANY_FIRSTY_INT__REG   BMG160_INT_STATUS2_ADDR

/**< 0th bit of Interrupt status 2 register */
#define BMG160_INT_STATUS2_ANY_FIRSTX_INT__POS   0
#define BMG160_INT_STATUS2_ANY_FIRSTX_INT__LEN   1
#define BMG160_INT_STATUS2_ANY_FIRSTX_INT__MSK   0x01
#define BMG160_INT_STATUS2_ANY_FIRSTX_INT__REG   BMG160_INT_STATUS2_ADDR

/**< 4 bits of Interrupt status 2 register */
#define BMG160_INT_STATUSTWO__POS   0
#define BMG160_INT_STATUSTWO__LEN   4
#define BMG160_INT_STATUSTWO__MSK   0x0F
#define BMG160_INT_STATUSTWO__REG   BMG160_INT_STATUS2_ADDR

/* Interrupt status 3 Register */
/**< 3th bit of Interrupt status 3 register */
#define BMG160_INT_STATUS3_HIGH_SIGN_INT__POS     3
#define BMG160_INT_STATUS3_HIGH_SIGN_INT__LEN     1
#define BMG160_INT_STATUS3_HIGH_SIGN_INT__MSK     0x08
#define BMG160_INT_STATUS3_HIGH_SIGN_INT__REG     BMG160_INT_STATUS3_ADDR

/**< 2th bit of Interrupt status 3 register */
#define BMG160_INT_STATUS3_HIGH_FIRSTZ_INT__POS   2
#define BMG160_INT_STATUS3_HIGH_FIRSTZ_INT__LEN   1
#define BMG160_INT_STATUS3_HIGH_FIRSTZ_INT__MSK   0x04
#define BMG160_INT_STATUS3_HIGH_FIRSTZ_INT__REG  BMG160_INT_STATUS3_ADDR

/**< 1st bit of Interrupt status 3 register */
#define BMG160_INT_STATUS3_HIGH_FIRSTY_INT__POS   1
#define BMG160_INT_STATUS3_HIGH_FIRSTY_INT__LEN   1
#define BMG160_INT_STATUS3_HIGH_FIRSTY_INT__MSK   0x02
#define BMG160_INT_STATUS3_HIGH_FIRSTY_INT__REG   BMG160_INT_STATUS3_ADDR

/**< 0th bit of Interrupt status 3 register */
#define BMG160_INT_STATUS3_HIGH_FIRSTX_INT__POS   0
#define BMG160_INT_STATUS3_HIGH_FIRSTX_INT__LEN   1
#define BMG160_INT_STATUS3_HIGH_FIRSTX_INT__MSK   0x01
#define BMG160_INT_STATUS3_HIGH_FIRSTX_INT__REG   BMG160_INT_STATUS3_ADDR

/**< LSB 4 bits of Interrupt status 3 register */
#define BMG160_INT_STATUSTHREE__POS   0
#define BMG160_INT_STATUSTHREE__LEN   4
#define BMG160_INT_STATUSTHREE__MSK   0x0F
#define BMG160_INT_STATUSTHREE__REG   BMG160_INT_STATUS3_ADDR

/* BMG160 FIFO Status Register */
/**< 7th bit of FIFO status Register */
#define BMG160_FIFO_STATUS_OVERRUN__POS         7
#define BMG160_FIFO_STATUS_OVERRUN__LEN         1
#define BMG160_FIFO_STATUS_OVERRUN__MSK         0x80
#define BMG160_FIFO_STATUS_OVERRUN__REG         BMG160_FIFO_STATUS_ADDR

/**< First 7 bits of FIFO status Register */
#define BMG160_FIFO_STATUS_FRAME_COUNTER__POS   0
#define BMG160_FIFO_STATUS_FRAME_COUNTER__LEN   7
#define BMG160_FIFO_STATUS_FRAME_COUNTER__MSK   0x7F
#define BMG160_FIFO_STATUS_FRAME_COUNTER__REG   BMG160_FIFO_STATUS_ADDR

/**< First 3 bits of range Registers */
#define BMG160_RANGE_ADDR_RANGE__POS           0
#define BMG160_RANGE_ADDR_RANGE__LEN           3
#define BMG160_RANGE_ADDR_RANGE__MSK           0x07
#define BMG160_RANGE_ADDR_RANGE__REG           BMG160_RANGE_ADDR

/**< Last bit of Bandwidth Registers */
#define BMG160_BW_ADDR_HIGH_RES__POS       7
#define BMG160_BW_ADDR_HIGH_RES__LEN       1
#define BMG160_BW_ADDR_HIGH_RES__MSK       0x80
#define BMG160_BW_ADDR_HIGH_RES__REG       BMG160_BW_ADDR

/**< First 3 bits of Bandwidth Registers */
#define BMG160_BW_ADDR__POS             0
#define BMG160_BW_ADDR__LEN             3
#define BMG160_BW_ADDR__MSK             0x07
#define BMG160_BW_ADDR__REG             BMG160_BW_ADDR

/**< 6th bit of Bandwidth Registers */
#define BMG160_BW_ADDR_IMG_STB__POS             6
#define BMG160_BW_ADDR_IMG_STB__LEN             1
#define BMG160_BW_ADDR_IMG_STB__MSK             0x40
#define BMG160_BW_ADDR_IMG_STB__REG             BMG160_BW_ADDR

/**< 5th and 7th bit of LPM1 Register */
#define BMG160_MODE_LPM1__POS             5
#define BMG160_MODE_LPM1__LEN             3
#define BMG160_MODE_LPM1__MSK             0xA0
#define BMG160_MODE_LPM1__REG             BMG160_MODE_LPM1_ADDR

/**< 1st to 3rd bit of LPM1 Register */
#define BMG160_MODELPM1_ADDR_SLEEPDUR__POS              1
#define BMG160_MODELPM1_ADDR_SLEEPDUR__LEN              3
#define BMG160_MODELPM1_ADDR_SLEEPDUR__MSK              0x0E
#define BMG160_MODELPM1_ADDR_SLEEPDUR__REG              BMG160_MODE_LPM1_ADDR

/**< 7th bit of Mode LPM2 Register */
#define BMG160_MODE_LPM2_ADDR_FAST_POWERUP__POS         7
#define BMG160_MODE_LPM2_ADDR_FAST_POWERUP__LEN         1
#define BMG160_MODE_LPM2_ADDR_FAST_POWERUP__MSK         0x80
#define BMG160_MODE_LPM2_ADDR_FAST_POWERUP__REG         BMG160_MODE_LPM2_ADDR

/**< 6th bit of Mode LPM2 Register */
#define BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING__POS      6
#define BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING__LEN      1
#define BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING__MSK      0x40
#define BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING__REG      BMG160_MODE_LPM2_ADDR

/**< 4th & 5th bit of Mode LPM2 Register */
#define BMG160_MODE_LPM2_ADDR_EXT_TRI_SEL__POS          4
#define BMG160_MODE_LPM2_ADDR_EXT_TRI_SEL__LEN          2
#define BMG160_MODE_LPM2_ADDR_EXT_TRI_SEL__MSK          0x30
#define BMG160_MODE_LPM2_ADDR_EXT_TRI_SEL__REG          BMG160_MODE_LPM2_ADDR

/**< 0th to 2nd bit of LPM2 Register */
#define BMG160_MODE_LPM2_ADDR_AUTOSLEEPDUR__POS  0
#define BMG160_MODE_LPM2_ADDR_AUTOSLEEPDUR__LEN  3
#define BMG160_MODE_LPM2_ADDR_AUTOSLEEPDUR__MSK  0x07
#define BMG160_MODE_LPM2_ADDR_AUTOSLEEPDUR__REG  BMG160_MODE_LPM2_ADDR

/**< 7th bit of HBW Register */
#define BMG160_RATED_HBW_ADDR_DATA_HIGHBW__POS         7
#define BMG160_RATED_HBW_ADDR_DATA_HIGHBW__LEN         1
#define BMG160_RATED_HBW_ADDR_DATA_HIGHBW__MSK         0x80
#define BMG160_RATED_HBW_ADDR_DATA_HIGHBW__REG         BMG160_RATED_HBW_ADDR

/**< 6th bit of HBW Register */
#define BMG160_RATED_HBW_ADDR_SHADOW_DIS__POS          6
#define BMG160_RATED_HBW_ADDR_SHADOW_DIS__LEN          1
#define BMG160_RATED_HBW_ADDR_SHADOW_DIS__MSK          0x40
#define BMG160_RATED_HBW_ADDR_SHADOW_DIS__REG          BMG160_RATED_HBW_ADDR

/**< 7th bit of Interrupt Enable 0 Registers */
#define BMG160_INT_ENABLE0_DATAEN__POS               7
#define BMG160_INT_ENABLE0_DATAEN__LEN               1
#define BMG160_INT_ENABLE0_DATAEN__MSK               0x80
#define BMG160_INT_ENABLE0_DATAEN__REG               BMG160_INT_ENABLE0_ADDR

/**< 6th bit of Interrupt Enable 0 Registers */
#define BMG160_INT_ENABLE0_FIFOEN__POS               6
#define BMG160_INT_ENABLE0_FIFOEN__LEN               1
#define BMG160_INT_ENABLE0_FIFOEN__MSK               0x40
#define BMG160_INT_ENABLE0_FIFOEN__REG               BMG160_INT_ENABLE0_ADDR

/**< 2nd bit of Interrupt Enable 0 Registers */
#define BMG160_INT_ENABLE0_AUTO_OFFSETEN__POS        2
#define BMG160_INT_ENABLE0_AUTO_OFFSETEN__LEN        1
#define BMG160_INT_ENABLE0_AUTO_OFFSETEN__MSK        0x04
#define BMG160_INT_ENABLE0_AUTO_OFFSETEN__REG        BMG160_INT_ENABLE0_ADDR

/**< 3rd bit of Interrupt Enable 1 Registers */
#define BMG160_INT_ENABLE1_IT2_OD__POS               3
#define BMG160_INT_ENABLE1_IT2_OD__LEN               1
#define BMG160_INT_ENABLE1_IT2_OD__MSK               0x08
#define BMG160_INT_ENABLE1_IT2_OD__REG               BMG160_INT_ENABLE1_ADDR

/**< 2nd bit of Interrupt Enable 1 Registers */
#define BMG160_INT_ENABLE1_IT2_LVL__POS              2
#define BMG160_INT_ENABLE1_IT2_LVL__LEN              1
#define BMG160_INT_ENABLE1_IT2_LVL__MSK              0x04
#define BMG160_INT_ENABLE1_IT2_LVL__REG              BMG160_INT_ENABLE1_ADDR

/**< 1st bit of Interrupt Enable 1 Registers */
#define BMG160_INT_ENABLE1_IT1_OD__POS               1
#define BMG160_INT_ENABLE1_IT1_OD__LEN               1
#define BMG160_INT_ENABLE1_IT1_OD__MSK               0x02
#define BMG160_INT_ENABLE1_IT1_OD__REG               BMG160_INT_ENABLE1_ADDR

/**< 0th bit of Interrupt Enable 1 Registers */
#define BMG160_INT_ENABLE1_IT1_LVL__POS              0
#define BMG160_INT_ENABLE1_IT1_LVL__LEN              1
#define BMG160_INT_ENABLE1_IT1_LVL__MSK              0x01
#define BMG160_INT_ENABLE1_IT1_LVL__REG              BMG160_INT_ENABLE1_ADDR

/**< 3rd bit of Interrupt MAP 0 Registers */
#define BMG160_INT_MAP_0_INT1_HIGH__POS            3
#define BMG160_INT_MAP_0_INT1_HIGH__LEN            1
#define BMG160_INT_MAP_0_INT1_HIGH__MSK            0x08
#define BMG160_INT_MAP_0_INT1_HIGH__REG            BMG160_INT_MAP_0_ADDR

/**< 1st bit of Interrupt MAP 0 Registers */
#define BMG160_INT_MAP_0_INT1_ANY__POS             1
#define BMG160_INT_MAP_0_INT1_ANY__LEN             1
#define BMG160_INT_MAP_0_INT1_ANY__MSK             0x02
#define BMG160_INT_MAP_0_INT1_ANY__REG             BMG160_INT_MAP_0_ADDR

/**< 7th bit of MAP_1Registers */
#define BMG160_MAP_1_INT2_DATA__POS                  7
#define BMG160_MAP_1_INT2_DATA__LEN                  1
#define BMG160_MAP_1_INT2_DATA__MSK                  0x80
#define BMG160_MAP_1_INT2_DATA__REG                  BMG160_INT_MAP_1_ADDR

/**< 6th bit of MAP_1Registers */
#define BMG160_MAP_1_INT2_FAST_OFFSET__POS           6
#define BMG160_MAP_1_INT2_FAST_OFFSET__LEN           1
#define BMG160_MAP_1_INT2_FAST_OFFSET__MSK           0x40
#define BMG160_MAP_1_INT2_FAST_OFFSET__REG           BMG160_INT_MAP_1_ADDR

/**< 5th bit of MAP_1Registers */
#define BMG160_MAP_1_INT2_FIFO__POS                  5
#define BMG160_MAP_1_INT2_FIFO__LEN                  1
#define BMG160_MAP_1_INT2_FIFO__MSK                  0x20
#define BMG160_MAP_1_INT2_FIFO__REG                  BMG160_INT_MAP_1_ADDR

/**< 4th bit of MAP_1Registers */
#define BMG160_MAP_1_INT2_AUTO_OFFSET__POS           4
#define BMG160_MAP_1_INT2_AUTO_OFFSET__LEN           1
#define BMG160_MAP_1_INT2_AUTO_OFFSET__MSK           0x10
#define BMG160_MAP_1_INT2_AUTO_OFFSET__REG           BMG160_INT_MAP_1_ADDR

/**< 3rd bit of MAP_1Registers */
#define BMG160_MAP_1_INT1_AUTO_OFFSET__POS           3
#define BMG160_MAP_1_INT1_AUTO_OFFSET__LEN           1
#define BMG160_MAP_1_INT1_AUTO_OFFSET__MSK           0x08
#define BMG160_MAP_1_INT1_AUTO_OFFSET__REG           BMG160_INT_MAP_1_ADDR

/**< 2nd bit of MAP_1Registers */
#define BMG160_MAP_1_INT1_FIFO__POS                  2
#define BMG160_MAP_1_INT1_FIFO__LEN                  1
#define BMG160_MAP_1_INT1_FIFO__MSK                  0x04
#define BMG160_MAP_1_INT1_FIFO__REG                  BMG160_INT_MAP_1_ADDR

/**< 1st bit of MAP_1Registers */
#define BMG160_MAP_1_INT1_FAST_OFFSET__POS           1
#define BMG160_MAP_1_INT1_FAST_OFFSET__LEN           1
#define BMG160_MAP_1_INT1_FAST_OFFSET__MSK           0x02
#define BMG160_MAP_1_INT1_FAST_OFFSET__REG           BMG160_INT_MAP_1_ADDR

/**< 0th bit of MAP_1Registers */
#define BMG160_MAP_1_INT1_DATA__POS                  0
#define BMG160_MAP_1_INT1_DATA__LEN                  1
#define BMG160_MAP_1_INT1_DATA__MSK                  0x01
#define BMG160_MAP_1_INT1_DATA__REG                  BMG160_INT_MAP_1_ADDR

/**< 3rd bit of Interrupt Map 2 Registers */
#define BMG160_INT_MAP_2_INT2_HIGH__POS            3
#define BMG160_INT_MAP_2_INT2_HIGH__LEN            1
#define BMG160_INT_MAP_2_INT2_HIGH__MSK            0x08
#define BMG160_INT_MAP_2_INT2_HIGH__REG            BMG160_INT_MAP_2_ADDR

/**< 1st bit of Interrupt Map 2 Registers */
#define BMG160_INT_MAP_2_INT2_ANY__POS             1
#define BMG160_INT_MAP_2_INT2_ANY__LEN             1
#define BMG160_INT_MAP_2_INT2_ANY__MSK             0x02
#define BMG160_INT_MAP_2_INT2_ANY__REG             BMG160_INT_MAP_2_ADDR

/**< 5th bit of Interrupt 0 Registers */
#define BMG160_INT_0_ADDR_SLOW_OFFSET_UNFILT__POS          5
#define BMG160_INT_0_ADDR_SLOW_OFFSET_UNFILT__LEN          1
#define BMG160_INT_0_ADDR_SLOW_OFFSET_UNFILT__MSK          0x20
#define BMG160_INT_0_ADDR_SLOW_OFFSET_UNFILT__REG          BMG160_INT_0_ADDR

/**< 3rd bit of Interrupt 0 Registers */
#define BMG160_INT_0_ADDR_HIGH_UNFILT_DATA__POS            3
#define BMG160_INT_0_ADDR_HIGH_UNFILT_DATA__LEN            1
#define BMG160_INT_0_ADDR_HIGH_UNFILT_DATA__MSK            0x08
#define BMG160_INT_0_ADDR_HIGH_UNFILT_DATA__REG            BMG160_INT_0_ADDR

/**< 1st bit of Interrupt 0 Registers */
#define BMG160_INT_0_ADDR_ANY_UNFILT_DATA__POS             1
#define BMG160_INT_0_ADDR_ANY_UNFILT_DATA__LEN             1
#define BMG160_INT_0_ADDR_ANY_UNFILT_DATA__MSK             0x02
#define BMG160_INT_0_ADDR_ANY_UNFILT_DATA__REG             BMG160_INT_0_ADDR

/**< 7th bit of INT_1  Registers */
#define BMG160_INT_1_ADDR_FAST_OFFSET_UNFILT__POS            7
#define BMG160_INT_1_ADDR_FAST_OFFSET_UNFILT__LEN            1
#define BMG160_INT_1_ADDR_FAST_OFFSET_UNFILT__MSK            0x80
#define BMG160_INT_1_ADDR_FAST_OFFSET_UNFILT__REG            BMG160_INT_1_ADDR

/**< First 7 bits of INT_1  Registers */
#define BMG160_INT_1_ADDR_ANY_TH__POS                       0
#define BMG160_INT_1_ADDR_ANY_TH__LEN                       7
#define BMG160_INT_1_ADDR_ANY_TH__MSK                       0x7F
#define BMG160_INT_1_ADDR_ANY_TH__REG                       BMG160_INT_1_ADDR

/**< Last 2 bits of INT 2Registers */
#define BMG160_INT_2_ADDR_AWAKE_DUR__POS          6
#define BMG160_INT_2_ADDR_AWAKE_DUR__LEN          2
#define BMG160_INT_2_ADDR_AWAKE_DUR__MSK          0xC0
#define BMG160_INT_2_ADDR_AWAKE_DUR__REG          BMG160_INT_2_ADDR

/**< 4th & 5th bit of INT 2Registers */
#define BMG160_INT_2_ADDR_ANY_DURSAMPLE__POS      4
#define BMG160_INT_2_ADDR_ANY_DURSAMPLE__LEN      2
#define BMG160_INT_2_ADDR_ANY_DURSAMPLE__MSK      0x30
#define BMG160_INT_2_ADDR_ANY_DURSAMPLE__REG      BMG160_INT_2_ADDR

/**< 2nd bit of INT 2Registers */
#define BMG160_INT_2_ADDR_ANY_EN_Z__POS           2
#define BMG160_INT_2_ADDR_ANY_EN_Z__LEN           1
#define BMG160_INT_2_ADDR_ANY_EN_Z__MSK           0x04
#define BMG160_INT_2_ADDR_ANY_EN_Z__REG           BMG160_INT_2_ADDR

/**< 1st bit of INT 2Registers */
#define BMG160_INT_2_ADDR_ANY_EN_Y__POS           1
#define BMG160_INT_2_ADDR_ANY_EN_Y__LEN           1
#define BMG160_INT_2_ADDR_ANY_EN_Y__MSK           0x02
#define BMG160_INT_2_ADDR_ANY_EN_Y__REG           BMG160_INT_2_ADDR

/**< 0th bit of INT 2Registers */
#define BMG160_INT_2_ADDR_ANY_EN_X__POS           0
#define BMG160_INT_2_ADDR_ANY_EN_X__LEN           1
#define BMG160_INT_2_ADDR_ANY_EN_X__MSK           0x01
#define BMG160_INT_2_ADDR_ANY_EN_X__REG           BMG160_INT_2_ADDR

/**< Last bit of INT 4 Registers */
#define BMG160_INT_4_FIFO_WM_EN__POS           7
#define BMG160_INT_4_FIFO_WM_EN__LEN           1
#define BMG160_INT_4_FIFO_WM_EN__MSK           0x80
#define BMG160_INT_4_FIFO_WM_EN__REG           BMG160_INT_4_ADDR

/**< Last bit of Reset Latch Registers */
#define BMG160_RST_LATCH_ADDR_RESET_INT__POS           7
#define BMG160_RST_LATCH_ADDR_RESET_INT__LEN           1
#define BMG160_RST_LATCH_ADDR_RESET_INT__MSK           0x80
#define BMG160_RST_LATCH_ADDR_RESET_INT__REG           BMG160_RST_LATCH_ADDR

/**< 6th bit of Reset Latch Registers */
#define BMG160_RST_LATCH_ADDR_OFFSET_RESET__POS        6
#define BMG160_RST_LATCH_ADDR_OFFSET_RESET__LEN        1
#define BMG160_RST_LATCH_ADDR_OFFSET_RESET__MSK        0x40
#define BMG160_RST_LATCH_ADDR_OFFSET_RESET__REG        BMG160_RST_LATCH_ADDR

/**< 4th bit of Reset Latch Registers */
#define BMG160_RST_LATCH_ADDR_LATCH_STATUS__POS        4
#define BMG160_RST_LATCH_ADDR_LATCH_STATUS__LEN        1
#define BMG160_RST_LATCH_ADDR_LATCH_STATUS__MSK        0x10
#define BMG160_RST_LATCH_ADDR_LATCH_STATUS__REG        BMG160_RST_LATCH_ADDR

/**< First 4 bits of Reset Latch Registers */
#define BMG160_RST_LATCH_ADDR_LATCH_INT__POS           0
#define BMG160_RST_LATCH_ADDR_LATCH_INT__LEN           4
#define BMG160_RST_LATCH_ADDR_LATCH_INT__MSK           0x0F
#define BMG160_RST_LATCH_ADDR_LATCH_INT__REG           BMG160_RST_LATCH_ADDR

/**< Last 2 bits of HIGH_TH_X Registers */
#define BMG160_HIGH_HY_X__POS        6
#define BMG160_HIGH_HY_X__LEN        2
#define BMG160_HIGH_HY_X__MSK        0xC0
#define BMG160_HIGH_HY_X__REG        BMG160_HIGH_TH_X_ADDR

/**< 5 bits of HIGH_TH_X Registers */
#define BMG160_HIGH_TH_X__POS        1
#define BMG160_HIGH_TH_X__LEN        5
#define BMG160_HIGH_TH_X__MSK        0x3E
#define BMG160_HIGH_TH_X__REG        BMG160_HIGH_TH_X_ADDR

/**< 0th bit of HIGH_TH_X Registers */
#define BMG160_HIGH_EN_X__POS        0
#define BMG160_HIGH_EN_X__LEN        1
#define BMG160_HIGH_EN_X__MSK        0x01
#define BMG160_HIGH_EN_X__REG        BMG160_HIGH_TH_X_ADDR

/**< Last 2 bits of HIGH_TH_Y Registers */
#define BMG160_HIGH_HY_Y__POS        6
#define BMG160_HIGH_HY_Y__LEN        2
#define BMG160_HIGH_HY_Y__MSK        0xC0
#define BMG160_HIGH_HY_Y__REG        BMG160_HIGH_TH_Y_ADDR

/**< 5 bits of HIGH_TH_Y Registers */
#define BMG160_HIGH_TH_Y__POS        1
#define BMG160_HIGH_TH_Y__LEN        5
#define BMG160_HIGH_TH_Y__MSK        0x3E
#define BMG160_HIGH_TH_Y__REG        BMG160_HIGH_TH_Y_ADDR

/**< 0th bit of HIGH_TH_Y Registers */
#define BMG160_HIGH_EN_Y__POS        0
#define BMG160_HIGH_EN_Y__LEN        1
#define BMG160_HIGH_EN_Y__MSK        0x01
#define BMG160_HIGH_EN_Y__REG        BMG160_HIGH_TH_Y_ADDR

/**< Last 2 bits of HIGH_TH_Z Registers */
#define BMG160_HIGH_HY_Z__POS        6
#define BMG160_HIGH_HY_Z__LEN        2
#define BMG160_HIGH_HY_Z__MSK        0xC0
#define BMG160_HIGH_HY_Z__REG        BMG160_HIGH_TH_Z_ADDR

/**< 5 bits of HIGH_TH_Z Registers */
#define BMG160_HIGH_TH_Z__POS        1
#define BMG160_HIGH_TH_Z__LEN        5
#define BMG160_HIGH_TH_Z__MSK        0x3E
#define BMG160_HIGH_TH_Z__REG        BMG160_HIGH_TH_Z_ADDR

/**< 0th bit of HIGH_TH_Z Registers */
#define BMG160_HIGH_EN_Z__POS        0
#define BMG160_HIGH_EN_Z__LEN        1
#define BMG160_HIGH_EN_Z__MSK        0x01
#define BMG160_HIGH_EN_Z__REG        BMG160_HIGH_TH_Z_ADDR

/**< Last 3 bits of INT OFF0 Registers */
#define BMG160_SLOW_OFFSET_TH__POS          6
#define BMG160_SLOW_OFFSET_TH__LEN          2
#define BMG160_SLOW_OFFSET_TH__MSK          0xC0
#define BMG160_SLOW_OFFSET_TH__REG          BMG160_SOC_ADDR

/**< 2  bits of INT OFF0 Registers */
#define BMG160_SLOW_OFFSET_DUR__POS         3
#define BMG160_SLOW_OFFSET_DUR__LEN         3
#define BMG160_SLOW_OFFSET_DUR__MSK         0x38
#define BMG160_SLOW_OFFSET_DUR__REG         BMG160_SOC_ADDR

/**< 2nd bit of INT OFF0 Registers */
#define BMG160_SLOW_OFFSET_EN_Z__POS        2
#define BMG160_SLOW_OFFSET_EN_Z__LEN        1
#define BMG160_SLOW_OFFSET_EN_Z__MSK        0x04
#define BMG160_SLOW_OFFSET_EN_Z__REG        BMG160_SOC_ADDR

/**< 1st bit of INT OFF0 Registers */
#define BMG160_SLOW_OFFSET_EN_Y__POS        1
#define BMG160_SLOW_OFFSET_EN_Y__LEN        1
#define BMG160_SLOW_OFFSET_EN_Y__MSK        0x02
#define BMG160_SLOW_OFFSET_EN_Y__REG        BMG160_SOC_ADDR

/**< 0th bit of INT OFF0 Registers */
#define BMG160_SLOW_OFFSET_EN_X__POS        0
#define BMG160_SLOW_OFFSET_EN_X__LEN        1
#define BMG160_SLOW_OFFSET_EN_X__MSK        0x01
#define BMG160_SLOW_OFFSET_EN_X__REG        BMG160_SOC_ADDR

/**< Last 2 bits of INT OFF1 Registers */
#define BMG160_AUTO_OFFSET_WL__POS        6
#define BMG160_AUTO_OFFSET_WL__LEN        2
#define BMG160_AUTO_OFFSET_WL__MSK        0xC0
#define BMG160_AUTO_OFFSET_WL__REG        BMG160_A_FOC_ADDR

/**< 2  bits of INT OFF1 Registers */
#define BMG160_FAST_OFFSET_WL__POS        4
#define BMG160_FAST_OFFSET_WL__LEN        2
#define BMG160_FAST_OFFSET_WL__MSK        0x30
#define BMG160_FAST_OFFSET_WL__REG        BMG160_A_FOC_ADDR

/**< 3nd bit of INT OFF1 Registers */
#define BMG160_FAST_OFFSET_EN__POS        3
#define BMG160_FAST_OFFSET_EN__LEN        1
#define BMG160_FAST_OFFSET_EN__MSK        0x08
#define BMG160_FAST_OFFSET_EN__REG        BMG160_A_FOC_ADDR

/**< 2nd bit of INT OFF1 Registers */
#define BMG160_FAST_OFFSET_EN_Z__POS      2
#define BMG160_FAST_OFFSET_EN_Z__LEN      1
#define BMG160_FAST_OFFSET_EN_Z__MSK      0x04
#define BMG160_FAST_OFFSET_EN_Z__REG      BMG160_A_FOC_ADDR

/**< 1st bit of INT OFF1 Registers */
#define BMG160_FAST_OFFSET_EN_Y__POS      1
#define BMG160_FAST_OFFSET_EN_Y__LEN      1
#define BMG160_FAST_OFFSET_EN_Y__MSK      0x02
#define BMG160_FAST_OFFSET_EN_Y__REG      BMG160_A_FOC_ADDR

/**< 0th bit of INT OFF1 Registers */
#define BMG160_FAST_OFFSET_EN_X__POS      0
#define BMG160_FAST_OFFSET_EN_X__LEN      1
#define BMG160_FAST_OFFSET_EN_X__MSK      0x01
#define BMG160_FAST_OFFSET_EN_X__REG      BMG160_A_FOC_ADDR

/**< 0 to 2 bits of INT OFF1 Registers */
#define BMG160_FAST_OFFSET_EN_XYZ__POS      0
#define BMG160_FAST_OFFSET_EN_XYZ__LEN      3
#define BMG160_FAST_OFFSET_EN_XYZ__MSK      0x07
#define BMG160_FAST_OFFSET_EN_XYZ__REG      BMG160_A_FOC_ADDR

/**< Last 4 bits of Trim NVM control Registers */
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_REMAIN__POS        4
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_REMAIN__LEN        4
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_REMAIN__MSK        0xF0
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_REMAIN__REG        \
BMG160_TRIM_NVM_CTRL_ADDR

/**< 3rd bit of Trim NVM control Registers */
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_LOAD__POS          3
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_LOAD__LEN          1
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_LOAD__MSK          0x08
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_LOAD__REG          \
BMG160_TRIM_NVM_CTRL_ADDR

/**< 2nd bit of Trim NVM control Registers */
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_RDY__POS           2
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_RDY__LEN           1
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_RDY__MSK           0x04
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_RDY__REG           \
BMG160_TRIM_NVM_CTRL_ADDR

 /**< 1st bit of Trim NVM control Registers */
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_TRIG__POS     1
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_TRIG__LEN     1
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_TRIG__MSK     0x02
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_TRIG__REG     \
BMG160_TRIM_NVM_CTRL_ADDR

/**< 0th bit of Trim NVM control Registers */
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_MODE__POS     0
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_MODE__LEN     1
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_MODE__MSK     0x01
#define BMG160_TRIM_NVM_CTRL_ADDR_NVM_PROG_MODE__REG     \
BMG160_TRIM_NVM_CTRL_ADDR

 /**< 2nd bit of SPI3 WDT Registers */
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_EN__POS      2
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_EN__LEN      1
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_EN__MSK      0x04
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_EN__REG      \
BMG160_BGW_SPI3_WDT_ADDR

 /**< 1st bit of SPI3 WDT Registers */
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_SEL__POS     1
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_SEL__LEN     1
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_SEL__MSK     0x02
#define BMG160_BGW_SPI3_WDT_ADDR_I2C_WDT_SEL__REG     \
BMG160_BGW_SPI3_WDT_ADDR

/**< 0th bit of SPI3 WDT Registers */
#define BMG160_BGW_SPI3_WDT_ADDR_SPI3__POS            0
#define BMG160_BGW_SPI3_WDT_ADDR_SPI3__LEN            1
#define BMG160_BGW_SPI3_WDT_ADDR_SPI3__MSK            0x01
#define BMG160_BGW_SPI3_WDT_ADDR_SPI3__REG            \
BMG160_BGW_SPI3_WDT_ADDR

/**< 4th bit of Self test Registers */
#define BMG160_SELF_TEST_ADDR_RATEOK__POS            4
#define BMG160_SELF_TEST_ADDR_RATEOK__LEN            1
#define BMG160_SELF_TEST_ADDR_RATEOK__MSK            0x10
#define BMG160_SELF_TEST_ADDR_RATEOK__REG            \
BMG160_SELF_TEST_ADDR

/**< 2nd bit of Self test Registers */
#define BMG160_SELF_TEST_ADDR_BISTFAIL__POS          2
#define BMG160_SELF_TEST_ADDR_BISTFAIL__LEN          1
#define BMG160_SELF_TEST_ADDR_BISTFAIL__MSK          0x04
#define BMG160_SELF_TEST_ADDR_BISTFAIL__REG          \
BMG160_SELF_TEST_ADDR

/**< 1st bit of Self test Registers */
#define BMG160_SELF_TEST_ADDR_BISTRDY__POS           1
#define BMG160_SELF_TEST_ADDR_BISTRDY__LEN           1
#define BMG160_SELF_TEST_ADDR_BISTRDY__MSK           0x02
#define BMG160_SELF_TEST_ADDR_BISTRDY__REG           \
BMG160_SELF_TEST_ADDR

/**< 0th bit of Self test Registers */
#define BMG160_SELF_TEST_ADDR_TRIGBIST__POS          0
#define BMG160_SELF_TEST_ADDR_TRIGBIST__LEN          1
#define BMG160_SELF_TEST_ADDR_TRIGBIST__MSK          0x01
#define BMG160_SELF_TEST_ADDR_TRIGBIST__REG          \
BMG160_SELF_TEST_ADDR

/**< 7th bit of FIFO CGF1 Registers */
#define BMG160_FIFO_CGF1_ADDR_TAG__POS     7
#define BMG160_FIFO_CGF1_ADDR_TAG__LEN     1
#define BMG160_FIFO_CGF1_ADDR_TAG__MSK     0x80
#define BMG160_FIFO_CGF1_ADDR_TAG__REG     BMG160_FIFO_CGF1_ADDR

/**< First 7 bits of FIFO CGF1 Registers */
#define BMG160_FIFO_CGF1_ADDR_WML__POS     0
#define BMG160_FIFO_CGF1_ADDR_WML__LEN     7
#define BMG160_FIFO_CGF1_ADDR_WML__MSK     0x7F
#define BMG160_FIFO_CGF1_ADDR_WML__REG     BMG160_FIFO_CGF1_ADDR

/**< Last 2 bits of FIFO CGF0 Addr Registers */
#define BMG160_FIFO_CGF0_ADDR_MODE__POS         6
#define BMG160_FIFO_CGF0_ADDR_MODE__LEN         2
#define BMG160_FIFO_CGF0_ADDR_MODE__MSK         0xC0
#define BMG160_FIFO_CGF0_ADDR_MODE__REG         BMG160_FIFO_CGF0_ADDR

/**< First 2 bits of FIFO CGF0 Addr Registers */
#define BMG160_FIFO_CGF0_ADDR_DATA_SEL__POS     0
#define BMG160_FIFO_CGF0_ADDR_DATA_SEL__LEN     2
#define BMG160_FIFO_CGF0_ADDR_DATA_SEL__MSK     0x03
#define BMG160_FIFO_CGF0_ADDR_DATA_SEL__REG     BMG160_FIFO_CGF0_ADDR

 /**< Last 2 bits of INL Offset MSB Registers */
#define BMG160_OFC1_ADDR_OFFSET_X__POS       6
#define BMG160_OFC1_ADDR_OFFSET_X__LEN       2
#define BMG160_OFC1_ADDR_OFFSET_X__MSK       0xC0
#define BMG160_OFC1_ADDR_OFFSET_X__REG       BMG160_OFC1_ADDR

/**< 3 bits of INL Offset MSB Registers */
#define BMG160_OFC1_ADDR_OFFSET_Y__POS       3
#define BMG160_OFC1_ADDR_OFFSET_Y__LEN       3
#define BMG160_OFC1_ADDR_OFFSET_Y__MSK       0x38
#define BMG160_OFC1_ADDR_OFFSET_Y__REG       BMG160_OFC1_ADDR

/**< First 3 bits of INL Offset MSB Registers */
#define BMG160_OFC1_ADDR_OFFSET_Z__POS       0
#define BMG160_OFC1_ADDR_OFFSET_Z__LEN       3
#define BMG160_OFC1_ADDR_OFFSET_Z__MSK       0x07
#define BMG160_OFC1_ADDR_OFFSET_Z__REG       BMG160_OFC1_ADDR

/**< 4 bits of Trim GP0 Registers */
#define BMG160_TRIM_GP0_ADDR_GP0__POS            4
#define BMG160_TRIM_GP0_ADDR_GP0__LEN            4
#define BMG160_TRIM_GP0_ADDR_GP0__MSK            0xF0
#define BMG160_TRIM_GP0_ADDR_GP0__REG            BMG160_TRIM_GP0_ADDR

/**< 2 bits of Trim GP0 Registers */
#define BMG160_TRIM_GP0_ADDR_OFFSET_X__POS       2
#define BMG160_TRIM_GP0_ADDR_OFFSET_X__LEN       2
#define BMG160_TRIM_GP0_ADDR_OFFSET_X__MSK       0x0C
#define BMG160_TRIM_GP0_ADDR_OFFSET_X__REG       BMG160_TRIM_GP0_ADDR

/**< 1st bit of Trim GP0 Registers */
#define BMG160_TRIM_GP0_ADDR_OFFSET_Y__POS       1
#define BMG160_TRIM_GP0_ADDR_OFFSET_Y__LEN       1
#define BMG160_TRIM_GP0_ADDR_OFFSET_Y__MSK       0x02
#define BMG160_TRIM_GP0_ADDR_OFFSET_Y__REG       BMG160_TRIM_GP0_ADDR

/**< First bit of Trim GP0 Registers */
#define BMG160_TRIM_GP0_ADDR_OFFSET_Z__POS       0
#define BMG160_TRIM_GP0_ADDR_OFFSET_Z__LEN       1
#define BMG160_TRIM_GP0_ADDR_OFFSET_Z__MSK       0x01
#define BMG160_TRIM_GP0_ADDR_OFFSET_Z__REG       BMG160_TRIM_GP0_ADDR

/* For Axis Selection   */
/**< It refers BMG160 X-axis */
#define BMG160_X_AXIS           0
/**< It refers BMG160 Y-axis */
#define BMG160_Y_AXIS           1
/**< It refers BMG160 Z-axis */
#define BMG160_Z_AXIS           2

/* For Mode Settings    */
#define BMG160_MODE_NORMAL              0
#define BMG160_MODE_DEEPSUSPEND         1
#define BMG160_MODE_SUSPEND             2
#define BMG160_MODE_FASTPOWERUP			3
#define BMG160_MODE_ADVANCEDPOWERSAVING 4

/* get bit slice  */
#define BMG160_GET_BITSLICE(regvar, bitname)\
((regvar & bitname##__MSK) >> bitname##__POS)

/* Set bit slice */
#define BMG160_SET_BITSLICE(regvar, bitname, val)\
((regvar&~bitname##__MSK)|((val<<bitname##__POS)&bitname##__MSK))
/* Constants */

#define BMG160_NULL                             0
/**< constant declaration of NULL */
#define BMG160_DISABLE                          0
/**< It refers BMG160 disable */
#define BMG160_ENABLE                           1
/**< It refers BMG160 enable */
#define BMG160_OFF                              0
/**< It refers BMG160 OFF state */
#define BMG160_ON                               1
/**< It refers BMG160 ON state  */

#define BMG160_TURN1                            0
/**< It refers BMG160 TURN1 */
#define BMG160_TURN2                            1
/**< It refers BMG160 TURN2 */

#define BMG160_INT1                             0
/**< It refers BMG160 INT1 */
#define BMG160_INT2                             1
/**< It refers BMG160 INT2 */

#define BMG160_SLOW_OFFSET                      0
/**< It refers BMG160 Slow Offset */
#define BMG160_AUTO_OFFSET                      1
/**< It refers BMG160 Auto Offset */
#define BMG160_FAST_OFFSET                      2
/**< It refers BMG160 Fast Offset */
#define BMG160_S_TAP                            0
/**< It refers BMG160 Single Tap */
#define BMG160_D_TAP                            1
/**< It refers BMG160 Double Tap */
#define BMG160_INT1_DATA                        0
/**< It refers BMG160 Int1 Data */
#define BMG160_INT2_DATA                        1
/**< It refers BMG160 Int2 Data */
#define BMG160_TAP_UNFILT_DATA                   0
/**< It refers BMG160 Tap unfilt data */
#define BMG160_HIGH_UNFILT_DATA                  1
/**< It refers BMG160 High unfilt data */
#define BMG160_CONST_UNFILT_DATA                 2
/**< It refers BMG160 Const unfilt data */
#define BMG160_ANY_UNFILT_DATA                   3
/**< It refers BMG160 Any unfilt data */
#define BMG160_SHAKE_UNFILT_DATA                 4
/**< It refers BMG160 Shake unfilt data */
#define BMG160_SHAKE_TH                         0
/**< It refers BMG160 Shake Threshold */
#define BMG160_SHAKE_TH2                        1
/**< It refers BMG160 Shake Threshold2 */
#define BMG160_AUTO_OFFSET_WL                   0
/**< It refers BMG160 Auto Offset word length */
#define BMG160_FAST_OFFSET_WL                   1
/**< It refers BMG160 Fast Offset word length */
#define BMG160_I2C_WDT_EN                       0
/**< It refers BMG160 I2C WDT En */
#define BMG160_I2C_WDT_SEL                      1
/**< It refers BMG160 I2C WDT Sel */
#define BMG160_EXT_MODE                         0
/**< It refers BMG160 Ext Mode */
#define BMG160_EXT_PAGE                         1
/**< It refers BMG160 Ext page */
#define BMG160_START_ADDR                       0
/**< It refers BMG160 Start Address */
#define BMG160_STOP_ADDR                        1
/**< It refers BMG160 Stop Address */
#define BMG160_SLOW_CMD                         0
/**< It refers BMG160 Slow Command */
#define BMG160_FAST_CMD                         1
/**< It refers BMG160 Fast Command */
#define BMG160_TRIM_VRA                         0
/**< It refers BMG160 Trim VRA */
#define BMG160_TRIM_VRD                         1
/**< It refers BMG160 Trim VRD */
#define BMG160_LOGBIT_EM                        0
/**< It refers BMG160 LogBit Em */
#define BMG160_LOGBIT_VM                        1
/**< It refers BMG160 LogBit VM */
#define BMG160_GP0                              0
/**< It refers BMG160 GP0 */
#define BMG160_GP1                              1
/**< It refers BMG160 GP1*/
#define BMG160_LOW_SPEED                        0
/**< It refers BMG160 Low Speed Oscillator */
#define BMG160_HIGH_SPEED                       1
/**< It refers BMG160 High Speed Oscillator */
#define BMG160_DRIVE_OFFSET_P                   0
/**< It refers BMG160 Drive Offset P */
#define BMG160_DRIVE_OFFSET_N                   1
/**< It refers BMG160 Drive Offset N */
#define BMG160_TEST_MODE_EN                     0
/**< It refers BMG160 Test Mode Enable */
#define BMG160_TEST_MODE_REG                    1
/**< It refers BMG160 Test Mode reg */
#define BMG160_IBIAS_DRIVE_TRIM                 0
/**< It refers BMG160 IBIAS Drive Trim */
#define BMG160_IBIAS_RATE_TRIM                  1
/**< It refers BMG160 IBIAS Rate Trim */
#define BMG160_BAA_MODE                         0
/**< It refers BMG160 BAA Mode Trim */
#define BMG160_BMA_MODE                         1
/**< It refers BMG160 BMA Mode Trim */
#define BMG160_PI_KP                            0
/**< It refers BMG160 PI KP */
#define BMG160_PI_KI                            1
/**< It refers BMG160 PI KI */


#define C_BMG160_SUCCESS						0
/**< It refers BMG160 operation is success */
#define C_BMG160_FAILURE						1
/**< It refers BMG160 operation is Failure */

#define BMG160_SPI_RD_MASK                      0x80
/**< Read mask **/
#define BMG160_READ_SET                         0x01
/**< Setting for rading data **/

#define BMG160_SHIFT_1_POSITION                 1
/**< Shift bit by 1 Position **/
#define BMG160_SHIFT_2_POSITION                 2
/**< Shift bit by 2 Position **/
#define BMG160_SHIFT_3_POSITION                 3
/**< Shift bit by 3 Position **/
#define BMG160_SHIFT_4_POSITION                 4
/**< Shift bit by 4 Position **/
#define BMG160_SHIFT_5_POSITION                 5
/**< Shift bit by 5 Position **/
#define BMG160_SHIFT_6_POSITION                 6
/**< Shift bit by 6 Position **/
#define BMG160_SHIFT_7_POSITION                 7
/**< Shift bit by 7 Position **/
#define BMG160_SHIFT_8_POSITION                 8
/**< Shift bit by 8 Position **/
#define BMG160_SHIFT_12_POSITION                12
/**< Shift bit by 12 Position **/

#define         C_BMG160_Null_U8X                              0
#define         C_BMG160_Zero_U8X                              0
#define         C_BMG160_One_U8X                               1
#define         C_BMG160_Two_U8X                               2
#define         C_BMG160_Three_U8X                             3
#define         C_BMG160_Four_U8X                              4
#define         C_BMG160_Five_U8X                              5
#define         C_BMG160_Six_U8X                               6
#define         C_BMG160_Seven_U8X                             7
#define         C_BMG160_Eight_U8X                             8
#define         C_BMG160_Nine_U8X                              9
#define         C_BMG160_Ten_U8X                               10
#define         C_BMG160_Eleven_U8X                            11
#define         C_BMG160_Twelve_U8X                            12
#define         C_BMG160_Thirteen_U8X                          13
#define         C_BMG160_Fifteen_U8X                           15
#define         C_BMG160_Sixteen_U8X                           16
#define         C_BMG160_TwentyTwo_U8X                         22
#define         C_BMG160_TwentyThree_U8X                       23
#define         C_BMG160_TwentyFour_U8X                        24
#define         C_BMG160_TwentyFive_U8X                        25
#define         C_BMG160_ThirtyTwo_U8X                         32
#define         C_BMG160_Hundred_U8X                           100
#define         C_BMG160_OneTwentySeven_U8X                    127
#define         C_BMG160_OneTwentyEight_U8X                    128
#define         C_BMG160_TwoFiftyFive_U8X                      255
#define         C_BMG160_TwoFiftySix_U16X                      256

#define         E_BMG160_NULL_PTR               (signed char)(-127)
#define         E_BMG160_COMM_RES               (signed char)(-1)
#define         E_BMG160_OUT_OF_RANGE           (signed char)(-2)

#define	C_BMG160_No_Filter_U8X			0
#define	C_BMG160_BW_230Hz_U8X			1
#define	C_BMG160_BW_116Hz_U8X			2
#define	C_BMG160_BW_47Hz_U8X			3
#define	C_BMG160_BW_23Hz_U8X			4
#define	C_BMG160_BW_12Hz_U8X			5
#define	C_BMG160_BW_64Hz_U8X			6
#define	C_BMG160_BW_32Hz_U8X			7

#define C_BMG160_No_AutoSleepDur_U8X	0
#define	C_BMG160_4ms_AutoSleepDur_U8X	1
#define	C_BMG160_5ms_AutoSleepDur_U8X	2
#define	C_BMG160_8ms_AutoSleepDur_U8X	3
#define	C_BMG160_10ms_AutoSleepDur_U8X	4
#define	C_BMG160_15ms_AutoSleepDur_U8X	5
#define	C_BMG160_20ms_AutoSleepDur_U8X	6
#define	C_BMG160_40ms_AutoSleepDur_U8X	7




#define BMG160_WR_FUNC_PTR int (*bus_write)\
(unsigned char, unsigned char *, unsigned char)
#define BMG160_RD_FUNC_PTR int (*bus_read)\
(unsigned char, unsigned char *, unsigned char)
#define BMG160_BRD_FUNC_PTR int (*burst_read)\
(unsigned char, unsigned char *, BMG160_S32)
#define BMG160_MDELAY_DATA_TYPE BMG160_U16


//#include "bmg160.h"


struct bmg160_data_t {
		BMG160_S16 datax;
		BMG160_S16 datay;
		BMG160_S16 dataz;
		char intstatus[5];
};



struct bmg160_t {
		unsigned char chip_id;
		BMG160_BRD_FUNC_PTR;
		BMG160_WR_FUNC_PTR;
		BMG160_RD_FUNC_PTR;
		void(*delay_msec)(BMG160_MDELAY_DATA_TYPE);
};

static struct bmg160_t *p_bmg160;


/*****************************************************************************
 * Description: *//**brief API Initialization routine
 *
 *
 *
 *
* \param bmg160_t *bmg160
 *      Pointer to a structure.
 *
 *       structure members are
 *
 *       unsigned char chip_id;
 *       unsigned char dev_addr;
 *       BMG160_BRD_FUNC_PTR;
 *       BMG160_WR_FUNC_PTR;
 *       BMG160_RD_FUNC_PTR;
 *       void(*delay_msec)( BMG160_MDELAY_DATA_TYPE );
 *
 *
 *
 *
 *
 *  \return result of communication routines
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_init(struct bmg160_t *bmg160)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;
	unsigned char a_data_u8r  = C_BMG160_Zero_U8X;
	p_bmg160 = bmg160;

	/*Read CHIP_ID */
	comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_CHIP_ID_ADDR, &a_data_u8r, 1);
	p_bmg160->chip_id = a_data_u8r;
	return comres;
}
BMG160_RETURN_FUNCTION_TYPE bmg160_get_dataXYZ(struct bmg160_data_t *data)
{
	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char a_data_u8r[6] = {0, 0, 0, 0, 0, 0};
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_RATE_X_LSB_VALUEX__REG, a_data_u8r, 6);
		a_data_u8r[0] =
		BMG160_GET_BITSLICE(a_data_u8r[0], BMG160_RATE_X_LSB_VALUEX);
		data->datax = (BMG160_S16)
		((((BMG160_S16)((signed char)a_data_u8r[1]))
		<< BMG160_SHIFT_8_POSITION) | (a_data_u8r[0]));
		a_data_u8r[2] = BMG160_GET_BITSLICE(a_data_u8r[2],
		BMG160_RATE_Y_LSB_VALUEY);
		data->datay = (BMG160_S16)
		((((BMG160_S16)((signed char)a_data_u8r[3]))
		<< BMG160_SHIFT_8_POSITION) | (a_data_u8r[2]));
		a_data_u8r[4] = BMG160_GET_BITSLICE(a_data_u8r[4],
		BMG160_RATE_Z_LSB_VALUEZ);
		data->dataz = (BMG160_S16)
		((((BMG160_S16)((signed char)a_data_u8r[5]))
		<< BMG160_SHIFT_8_POSITION) | (a_data_u8r[4]));
	}
	return comres;
}


/*****************************************************************************
 * Description: *//**brief This API is used to get the operating modes of the
 * sensor
 *
 *
 *
 *
 *\param unsigned char * mode : Address of mode
 *                       0 -> NORMAL
 *                       1 -> SUSPEND
 *                       2 -> DEEP SUSPEND
 *						 3 -> FAST POWERUP
 *						 4 -> ADVANCED POWERSAVING
 *
 *
 *  \return
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_get_mode(unsigned char *mode)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;
	unsigned char data1 = C_BMG160_Zero_U8X;
	unsigned char data2 = C_BMG160_Zero_U8X;
	unsigned char data3 = C_BMG160_Zero_U8X;
	if (p_bmg160 == C_BMG160_Zero_U8X) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_MODE_LPM1_ADDR, &data1, C_BMG160_One_U8X);
		comres += p_bmg160->BMG160_BUS_READ_FUNC(BMG160_MODE_LPM2_ADDR, &data2, C_BMG160_One_U8X);
		data1  = (data1 & 0xA0) >> 5;
		data3  = (data2 & 0x40) >> 6;
		data2  = (data2 & 0x80) >> 7;
		if (data3 == 0x01) {
			*mode  = BMG160_MODE_ADVANCEDPOWERSAVING;
		} else {
			if ((data1 == 0x00) && (data2 == 0x00)) {
				*mode  = BMG160_MODE_NORMAL;
				} else {
				if ((data1 == 0x01) || (data1 == 0x05)) {
					*mode  = BMG160_MODE_DEEPSUSPEND;
					} else {
					if ((data1 == 0x04) &&
					(data2 == 0x00)) {
						*mode  = BMG160_MODE_SUSPEND;
					} else {
					if ((data1 == 0x04) &&
						(data2 == 0x01))
							*mode  =
							BMG160_MODE_FASTPOWERUP;
						}
					}
				}
			}
		}
	return comres;
}
/* Compiler Switch if applicable
#ifdef

#endif
*/
/*****************************************************************************
 * Description: *//**brief This API is used to set the operating Modes of the
 * sensor
 *
 *
 *
 *
 *\param unsigned char Mode
 *                       0 -> NORMAL
 *                       1 -> DEEPSUSPEND
 *                       2 -> SUSPEND
 *						 3 -> Fast Powerup
 *						 4 -> Advance Powerup
 *  \return communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_set_mode(unsigned char mode)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;
	unsigned char data1 = C_BMG160_Zero_U8X;
	unsigned char data2 = C_BMG160_Zero_U8X;
	unsigned char data3 = C_BMG160_Zero_U8X;
	if (p_bmg160 == C_BMG160_Zero_U8X) {
		return  E_BMG160_NULL_PTR;
	} else {
		if (mode < C_BMG160_Five_U8X) {
			comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_MODE_LPM1_ADDR, &data1, C_BMG160_One_U8X);
			comres += p_bmg160->BMG160_BUS_READ_FUNC(BMG160_MODE_LPM2_ADDR, &data2, C_BMG160_One_U8X);
			switch (mode) {
			case BMG160_MODE_NORMAL:
				data1  = BMG160_SET_BITSLICE(data1,
				BMG160_MODE_LPM1, C_BMG160_Zero_U8X);
				data2  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_FAST_POWERUP,
				C_BMG160_Zero_U8X);
				data3  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING,
				C_BMG160_Zero_U8X);
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM1_ADDR, &data1, C_BMG160_One_U8X);
				p_bmg160->delay_msec(1);/*A minimum delay of atleast 450us is required for Multiple write.*/
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM2_ADDR, &data3, C_BMG160_One_U8X);
				break;
			case BMG160_MODE_DEEPSUSPEND:
				data1  = BMG160_SET_BITSLICE(data1,
				BMG160_MODE_LPM1, C_BMG160_One_U8X);
				data2  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_FAST_POWERUP,
				C_BMG160_Zero_U8X);
				data3  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING,
				C_BMG160_Zero_U8X);
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM1_ADDR, &data1, C_BMG160_One_U8X);
				p_bmg160->delay_msec(1);/*A minimum delay of atleast 450us is required for Multiple write.*/
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM2_ADDR, &data3, C_BMG160_One_U8X);
				break;
			case BMG160_MODE_SUSPEND:
				data1  = BMG160_SET_BITSLICE(data1,
				BMG160_MODE_LPM1, C_BMG160_Four_U8X);
				data2  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_FAST_POWERUP,
				C_BMG160_Zero_U8X);
				data3  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING,
				C_BMG160_Zero_U8X);
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM1_ADDR, &data1, C_BMG160_One_U8X);
				p_bmg160->delay_msec(1);/*A minimum delay of atleast 450us is required for Multiple write.*/
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM2_ADDR, &data3, C_BMG160_One_U8X);
				break;
			case BMG160_MODE_FASTPOWERUP:
				data1  = BMG160_SET_BITSLICE(data1,
				BMG160_MODE_LPM1, C_BMG160_Four_U8X);
				data2  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_FAST_POWERUP,
				C_BMG160_One_U8X);
				data3  = BMG160_SET_BITSLICE(data2,
				BMG160_MODE_LPM2_ADDR_ADV_POWERSAVING,
				C_BMG160_Zero_U8X);
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM1_ADDR, &data1, C_BMG160_One_U8X);
				p_bmg160->delay_msec(1);/*A minimum delay of atleast 450us is required for Multiple write.*/
				comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_MODE_LPM2_ADDR, &data3, C_BMG160_One_U8X);
				break;
			}
		} else {
		comres = E_BMG160_OUT_OF_RANGE;
		}
	}
	return comres;
}

/*****************************************************************************
 * Description: *//**brief This API reads the range from register 0x0Fh of
 * (0 to 2) bits
 *
 *
 *
 *
 *\param unsigned char *range
 *      Range[0....7]
 *      0 2000/s
 *      1 1000/s
 *      2 500/s
 *      3 250/s
 *      4 125/s
 *
 *
 *
 *
 *
 *  \return communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_get_range_reg(unsigned char *range)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_RANGE_ADDR_RANGE__REG, &v_data_u8r, 1);
		*range =
		BMG160_GET_BITSLICE(v_data_u8r, BMG160_RANGE_ADDR_RANGE);
	}
	return comres;
}
/* Compiler Switch if applicable
#ifdef

#endif
*/
/*****************************************************************************
 * Description: *//**brief This API sets the range register 0x0Fh
 * (0 to 2 bits)
 *
 *
 *
 *
 *\param unsigned char range
 *
 *      Range[0....7]
 *      0 2000/s
 *      1 1000/s
 *      2 500/s
 *      3 250/s
 *      4 125/s
 *
 *
 *
 *
 *  \return Communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_set_range_reg(unsigned char range)
{
	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		if (range < C_BMG160_Five_U8X) {
			comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_RANGE_ADDR_RANGE__REG, &v_data_u8r, 1);
			v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
			BMG160_RANGE_ADDR_RANGE,
			range);
			comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_RANGE_ADDR_RANGE__REG, &v_data_u8r, 1);
		} else {
			comres = E_BMG160_OUT_OF_RANGE;
		}
	}
	return comres;
}
/*****************************************************************************
 * Description: *//**brief This API reads the bandwidth register of 0x10h 0 to
 *  3 bits
 *
 *
 *
 *
* \param unsigned char *bandwidth
 *              pointer to a variable passed as a parameter
 *
 *              0 no filter(523 Hz)
 *              1 230Hz
 *              2 116Hz
 *              3 47Hz
 *              4 23Hz
 *              5 12Hz
 *              6 64Hz
 *              7 32Hz
 *
 *
 *
 *  \return communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_get_bw(unsigned char *bandwidth)
{
	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_BW_ADDR__REG, &v_data_u8r, 1);
		*bandwidth = BMG160_GET_BITSLICE(v_data_u8r,
			BMG160_BW_ADDR);
	}
	return comres;
}
/* Compiler Switch if applicable
#ifdef

#endif
*/
/*****************************************************************************
 * Description: *//**brief This API writes the Bandwidth register (0x10h of 0
 * to 3 bits)
 *
 *
 *
 *
 *\param unsigned char bandwidth,
 *              The bandwidth to be set passed as a parameter
 *
 *              0 no filter(523 Hz)
 *              1 230Hz
 *              2 116Hz
 *              3 47Hz
 *              4 23Hz
 *              5 12Hz
 *              6 64Hz
 *              7 32Hz
 *
 *
 *
 *
 *  \return communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_set_bw(unsigned char bandwidth)
{
	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	unsigned char v_mode_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		if (bandwidth < C_BMG160_Eight_U8X) {
			bmg160_get_mode(&v_mode_u8r);
			comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_BW_ADDR__REG, &v_data_u8r, 1);
			v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
				BMG160_BW_ADDR, bandwidth);
			comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_BW_ADDR__REG, &v_data_u8r, 1);
		} else {
			comres = E_BMG160_OUT_OF_RANGE;
		}
	}
	return comres;
}
/*****************************************************************************
 * Description: *//**brief This API is used to set data enable data
 *
 *
 *
 *
 *  \param unsigned char data_en:
 *          Value to be written passed as a \parameter
 *           0 --> Disable
 *           1 --> Enable
 *
 *
 *
 *  \return communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_set_data_en(unsigned char data_en)
{
	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_INT_ENABLE0_DATAEN__REG, &v_data_u8r, 1);
		v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
		BMG160_INT_ENABLE0_DATAEN, data_en);
		comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_INT_ENABLE0_DATAEN__REG, &v_data_u8r, 1);
	}
	return comres;
}


/*****************************************************************************
 * Description: *//**brief This API is used to set the output type status
 *
 *
 *
 *
 *  \param unsigned char channel,unsigned char *int_od
 *                  BMG160_INT1    ->   0
 *                  BMG160_INT2    ->   1
 *                  int_od : open drain   ->   1
 *                           push pull    ->   0
 *
 *
 *
 *  \return communication results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_set_int_od(unsigned char param,
unsigned char int_od)
{
	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		switch (param) {
		case BMG160_INT1:
			comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_INT_ENABLE1_IT1_OD__REG, &v_data_u8r, 1);
			v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
			BMG160_INT_ENABLE1_IT1_OD, int_od);
			comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_INT_ENABLE1_IT1_OD__REG, &v_data_u8r, 1);
			break;
		case BMG160_INT2:
			comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_INT_ENABLE1_IT2_OD__REG, &v_data_u8r, 1);
			v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
			BMG160_INT_ENABLE1_IT2_OD, int_od);
			comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_INT_ENABLE1_IT2_OD__REG, &v_data_u8r, 1);
			break;
		default:
			comres = E_BMG160_OUT_OF_RANGE;
			break;
		}
	}
	return comres;
}
/*****************************************************************************
 * Description: *//**brief Reads interrupt fifo status register byte from 0Eh
 *
 *
 *
 *
 *  \param
 *      unsigned char *fifo_framecount: Address of FIFO status register
 *
 *
 *  \return
 *      Result of bus communication function
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/

BMG160_RETURN_FUNCTION_TYPE bmg160_get_fifo_framecount(
unsigned char *fifo_framecount)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_FIFO_STATUS_FRAME_COUNTER__REG, &v_data_u8r, 1);
		*fifo_framecount = BMG160_GET_BITSLICE(v_data_u8r,
		BMG160_FIFO_STATUS_FRAME_COUNTER);
	}
	return comres;
}
/*****************************************************************************
 * Description: *//**brief This API is used to get the status of fifo mode
 *
 *
 *
 *
 *\param unsigned char *mode : Address of mode
 *                         fifo_mode  0 --> Bypass
 *                         1 --> FIFO
 *                         2 --> Stream
 *                         3 --> Reserved
 *
 *
 *
 *
 *  \return
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 * Remarks:
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_get_fifo_mode(unsigned char *mode)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_FIFO_CGF0_ADDR_MODE__REG, &v_data_u8r, 1);
		*mode = BMG160_GET_BITSLICE(v_data_u8r,
			BMG160_FIFO_CGF0_ADDR_MODE);
	}
	return comres;
}
/* Compiler Switch if applicable
#ifdef

#endif
*/
/*****************************************************************************
 * Description: *//**brief This API is used set to FIFO mode
 *
 *
 *
 *
* \param             0 --> BYPASS
 *                      1 --> FIFO
 *                      2 --> STREAM
 *
 *
 *  \return Communication Results
 *
 *
 *****************************************************************************/
/* Scheduling:
 *
 *
 *
 * Usage guide:
 *
 *
 *
 *****************************************************************************/
BMG160_RETURN_FUNCTION_TYPE bmg160_set_fifo_mode(unsigned char mode)
{
	int comres = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		if (mode < C_BMG160_Four_U8X) {
			comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_FIFO_CGF0_ADDR_MODE__REG, &v_data_u8r, 1);
			v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
			BMG160_FIFO_CGF0_ADDR_MODE, mode);
			v_data_u8r |= 0xc;
			comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_FIFO_CGF0_ADDR_MODE__REG, &v_data_u8r, 1);
		} else {
			comres = E_BMG160_OUT_OF_RANGE;
		}
	}
	return comres;
}


struct bmg_client_data {
	struct bmg160_t device;
	struct i2c_client *client;
	struct input_dev *input;
	struct regulator *v1p8_reg;
	struct regulator *v2p8_reg;
};

static struct i2c_client *bmg_client;
/* i2c operation for API */
static void bmg_i2c_delay(BMG160_U16 msec);
static int bmg_i2c_read(struct i2c_client *client, u8 reg_addr,
		u8 *data, u8 len);
static int bmg_i2c_write(struct i2c_client *client, u8 reg_addr,
		u8 *data, u8 len);

static int bmg_check_chip_id(struct i2c_client *client);


static int bmg160_set_newdata(unsigned char int_newdata)
{


	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;
	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;
	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
		comres = p_bmg160->BMG160_BUS_READ_FUNC(BMG160_INT_MAP_1_INT1DATA__REG , &v_data_u8r, 1);
		v_data_u8r = BMG160_SET_BITSLICE(v_data_u8r,
		BMG160_INT_MAP_1_INT1DATA , int_newdata);
		comres += p_bmg160->BMG160_BUS_WRITE_FUNC(BMG160_INT_MAP_1_INT1DATA__REG, &v_data_u8r, 1);
	}
	return comres;

}


static int bmg_check_chip_id(struct i2c_client *client)
{
	int err = -1;
	u8 chip_id = 0;
	u8 read_count = 0;

	while (read_count++ < CHECK_CHIP_ID_TIME_MAX) {
		bmg_i2c_read(client, BMG_REG_NAME(CHIP_ID_ADDR), &chip_id, 1);
		dev_info(&client->dev, "read chip id result: %#x", chip_id);

		if ((chip_id & 0xff) != SENSOR_CHIP_ID_BMG) {
			mdelay(1);
		} else {
			err = 0;
			break;
		}
	}
	return err;
}

static void bmg_i2c_delay(BMG160_U16 msec)
{
	mdelay(msec);
}

/*i2c read routine for API*/
static int bmg_i2c_read(struct i2c_client *client, u8 reg_addr,
		u8 *data, u8 len)
{
#if !defined BMG_USE_BASIC_I2C_FUNC
	s32 dummy;
	if (NULL == client)
		return -1;

	while (0 != len--) {
#ifdef BMG_SMBUS
		dummy = i2c_smbus_read_byte_data(client, reg_addr);
		if (dummy < 0) {
			dev_err(&client->dev, "i2c bus read error");
			return -1;
		}
		*data = (u8)(dummy & 0xff);
#else
		dummy = i2c_master_send(client, (char *)&reg_addr, 1);
		if (dummy < 0)
			return -1;

		dummy = i2c_master_recv(client, (char *)data, 1);
		if (dummy < 0)
			return -1;
#endif
		reg_addr++;
		data++;
	}
	return 0;
#else
	int retry;

	struct i2c_msg msg[] = {
		{
			.addr = client->addr,
			.flags = 0,
			.len = 1,
			.buf = &reg_addr,
		},

		{
			.addr = client->addr,
			.flags = I2C_M_RD,
			.len = len,
			.buf = data,
		},
	};

	for (retry = 0; retry < BMG_MAX_RETRY_I2C_XFER; retry++) {
		if (i2c_transfer(client->adapter, msg, ARRAY_SIZE(msg)) > 0)
			break;
		else
			mdelay(BMG_I2C_WRITE_DELAY_TIME);
	}

	if (BMG_MAX_RETRY_I2C_XFER <= retry) {
		dev_err(&client->dev, "I2C xfer error");
		return -EIO;
	}

	return 0;
#endif
}

static int bmg_i2c_burst_read(struct i2c_client *client, u8 reg_addr,
		u8 *data, u16 len)
{
	int retry;

	struct i2c_msg msg[] = {
		{
			.addr = client->addr,
			.flags = 0,
			.len = 1,
			.buf = &reg_addr,
		},

		{
			.addr = client->addr,
			.flags = I2C_M_RD,
			.len = len,
			.buf = data,
		},
	};

	for (retry = 0; retry < BMG_MAX_RETRY_I2C_XFER; retry++) {
		if (i2c_transfer(client->adapter, msg, ARRAY_SIZE(msg)) > 0)
			break;
		else
			mdelay(BMG_I2C_WRITE_DELAY_TIME);
	}

	if (BMG_MAX_RETRY_I2C_XFER <= retry) {
		dev_err(&client->dev, "I2C xfer error");
		return -EIO;
	}

	return 0;
}

/*i2c write routine for */
static int bmg_i2c_write(struct i2c_client *client, u8 reg_addr,
		u8 *data, u8 len)
{
#if !defined BMG_USE_BASIC_I2C_FUNC
	s32 dummy;

#ifndef BMG_SMBUS
	u8 buffer[2];
#endif

	if (NULL == client)
		return -EPERM;

	while (0 != len--) {
#ifdef BMG_SMBUS
		dummy = i2c_smbus_write_byte_data(client, reg_addr, *data);
#else
		buffer[0] = reg_addr;
		buffer[1] = *data;
		dummy = i2c_master_send(client, (char *)buffer, 2);
#endif
		reg_addr++;
		data++;
		if (dummy < 0) {
			dev_err(&client->dev, "error writing i2c bus");
			return -EPERM;
		}

	}
	return 0;
#else
	u8 buffer[2];
	int retry;
	struct i2c_msg msg[] = {
		{
		 .addr = client->addr,
		 .flags = 0,
		 .len = 2,
		 .buf = buffer,
		 },
	};

	while (0 != len--) {
		buffer[0] = reg_addr;
		buffer[1] = *data;
		for (retry = 0; retry < BMG_MAX_RETRY_I2C_XFER; retry++) {
			if (i2c_transfer(client->adapter, msg,
						ARRAY_SIZE(msg)) > 0) {
				break;
			} else {
				mdelay(BMG_I2C_WRITE_DELAY_TIME);
			}
		}
		if (BMG_MAX_RETRY_I2C_XFER <= retry) {
			dev_err(&client->dev, "I2C xfer error");
			return -EIO;
		}
		reg_addr++;
		data++;
	}

	return 0;
#endif
}

static int bmg_i2c_read_wrapper(u8 reg_addr, u8 *data, u8 len)
{
	int err;
	err = bmg_i2c_read(bmg_client, reg_addr, data, len);
	return err;
}

static int bmg_i2c_write_wrapper(u8 reg_addr, u8 *data, u8 len)
{
	int err;
	err = bmg_i2c_write(bmg_client, reg_addr, data, len);
	return err;
}

static int bmg160_power_ctrl(struct i2c_client *client, int flag)
{
	struct bmg_client_data *client_data =
		(struct bmg_client_data *)i2c_get_clientdata(client);
	int ret = -1;

	if (v2p8_gpio == V2P8_GPIO_UNSET) {
		v2p8_gpio = gmin_get_var_int(NULL, "V2P8GPIO", V2P8_GPIO_NONE);
		if (v2p8_gpio != V2P8_GPIO_NONE) {
			pr_info("atomisp_gmin_platform: 2.8v power on GPIO %d\n",
				v2p8_gpio);
			ret = gpio_request(v2p8_gpio, "camera_v2p8");
			if (!ret)
				ret = gpio_direction_output(v2p8_gpio, 0);
			if (ret)
				pr_err("V2P8 GPIO initialization failed\n");
		}
	}

	if (v1p8_gpio == V1P8_GPIO_UNSET) {
		v1p8_gpio = gmin_get_var_int(NULL, "V1P8GPIO", V1P8_GPIO_NONE);
		if (v1p8_gpio != V1P8_GPIO_NONE) {
			pr_info("atomisp_gmin_platform: 1.8v power on GPIO %d\n",
				v1p8_gpio);
			ret = gpio_request(v1p8_gpio, "camera_v1p8");
			if (!ret)
				ret = gpio_direction_output(v1p8_gpio, 0);
			if (ret)
				pr_err("V1P8 GPIO initialization failed\n");
		}
	}


	if (flag)
	{

		if (v2p8_gpio != V2P8_GPIO_NONE)
			gpio_set_value(v2p8_gpio, flag);
		regulator_enable(client_data->v2p8_reg);


		usleep_range(5000, 6000);

		if (v1p8_gpio != V1P8_GPIO_NONE)
			gpio_set_value(v1p8_gpio, flag);
		regulator_enable(client_data->v1p8_reg);
		usleep_range(5000, 6000);
	}
	else
	{
		if (v1p8_gpio != V1P8_GPIO_NONE)
			gpio_set_value(v1p8_gpio, flag);
		regulator_disable(client_data->v1p8_reg);

		usleep_range(5000, 6000);

		if (v2p8_gpio != V2P8_GPIO_NONE)
			gpio_set_value(v2p8_gpio, flag);
		regulator_disable(client_data->v2p8_reg);

	}
	return 0;
}

#if 0
static int bmg_set_soft_reset(struct i2c_client *client)
{
	int err = 0;
	unsigned char data = BMG_SOFT_RESET_VALUE;
	err = bmg_i2c_write(client, BMG160_BGW_SOFTRESET_ADDR, &data, 1);
	return err;
}
#endif
static ssize_t bmg_show_chip_id(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", SENSOR_CHIP_ID_BMG);
}

static ssize_t bmg_show_enable(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int ret;
	u8 op_mode = 0xff;
	int enable;

	BMG_CALL_API(get_mode)(&op_mode);

	if (op_mode == 0)
		enable = 1;
	else
		enable = 0;

	ret = sprintf(buf, "%d\n", enable);

	return ret;
}




static ssize_t bmg_enable_by_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int ret,err;
	unsigned char range = 1;
	bmg160_set_int_od(BMG160_INT1,0);
	// configure data_ready interrupt on int_3 signal
	bmg160_set_newdata(1);

	// enable data_ready interrupt.
	bmg160_set_data_en(1);
	bmg160_set_bw(C_BMG160_BW_32Hz_U8X);

	BMG_CALL_API(set_range_reg)(range);

	BMG_CALL_API(set_fifo_mode)(2);

	//	tmp = i2c_smbus_read_byte_data(client_data->client,0x34);
	//	tmp|=0x6;
	//	i2c_smbus_write_byte_data(client_data->client,0x34, tmp);

	err = BMG_CALL_API(set_mode)(0);

	if (err)
		ret = sprintf(buf, "e\n");
	else
		ret = sprintf(buf, "%d\n", 1);

	return ret;
}

static ssize_t bmg_disable_by_read(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int ret,err;

	err = BMG_CALL_API(set_mode)(3);
	if (err)
		ret = sprintf(buf, "e\n");
	else
		ret = sprintf(buf, "%d\n", 0);

	return ret;
}



static ssize_t bmg_store_enable(struct device *dev,
		struct device_attribute *attr,
		const char *buf, size_t count)
{
	int err;

	long enable;
	unsigned char range = 1;

	err = kstrtoul(buf, 10, &enable);
	if (err)
		return err;


	if (enable !=0)
	{
		bmg160_set_int_od(BMG160_INT1,0);
		// configure data_ready interrupt on int_3 signal
		bmg160_set_newdata(1);

		// enable data_ready interrupt.
		bmg160_set_data_en(1);
		bmg160_set_bw(C_BMG160_BW_32Hz_U8X);

		BMG_CALL_API(set_range_reg)(range);

		BMG_CALL_API(set_fifo_mode)(2);

		//	tmp = i2c_smbus_read_byte_data(client_data->client,0x34);
		//	tmp|=0x6;
		//	i2c_smbus_write_byte_data(client_data->client,0x34, tmp);

		err = BMG_CALL_API(set_mode)(0);
	}
	else
	{
		err = BMG_CALL_API(set_mode)(3);
	}


	if (err)
		return err;
	else
		return count;
}



static ssize_t bmg_show_value(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int count;

	struct bmg160_data_t value_data;
	BMG_CALL_API(get_dataXYZ)(&value_data);

	count = sprintf(buf, "%hd %hd %hd\n",
				value_data.datax,
				value_data.datay,
				value_data.dataz);

	return count;
}


static ssize_t bmg_show_fifo_framecount(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int err;
	unsigned char fifo_framecount;
	BMG_CALL_API(get_fifo_framecount)(&fifo_framecount);
	err = sprintf(buf, "%d\n", fifo_framecount);
	return err;
}




static ssize_t bmg_show_fifo_data_frame(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int i, len;
	signed char fifo_data_out[MAX_FIFO_F_LEVEL * MAX_FIFO_F_BYTES] = {0};
	unsigned char f_len = 6;
	struct input_dev *input = to_input_dev(dev);
	struct bmg_client_data *client_data = input_get_drvdata(input);
	struct bmg160_data_t gyro_lsb;

	int err = -EIO;

	unsigned char fifo_framecount = 0 ;
	BMG_CALL_API(get_fifo_framecount)(&fifo_framecount);


	if (fifo_framecount ==0)
		return err;


//	if (fifo_framecount < 2)
//		return 0;

//	fifo_framecount-=1;


	bmg_i2c_burst_read(client_data->client, BMG160_FIFO_DATA_ADDR,
			fifo_data_out, fifo_framecount * f_len);
	err=0;

	for (i = 0; i < fifo_framecount; i++) {
		gyro_lsb.datax =
		((unsigned char)fifo_data_out[i * f_len + 1] << 8
			| (unsigned char)fifo_data_out[i * f_len + 0]);

		gyro_lsb.datay =
		((unsigned char)fifo_data_out[i * f_len + 3] << 8
			| (unsigned char)fifo_data_out[i * f_len + 2]);

		gyro_lsb.dataz =
		((unsigned char)fifo_data_out[i * f_len + 5] << 8
			| (unsigned char)fifo_data_out[i * f_len + 4]);

		//bmg160_remap_sensor_data(&gyro_lsb, client_data);
		len = sprintf(buf, "%hd %hd %hd ",
			gyro_lsb.datax, gyro_lsb.datay, gyro_lsb.dataz);
		buf += len;
		err += len;
	}

	return err;
}


BMG160_RETURN_FUNCTION_TYPE bmg160_store_reg(unsigned char reg, unsigned char value)
{
	BMG160_RETURN_FUNCTION_TYPE comres = C_BMG160_Zero_U8X;

	unsigned char v_data_u8r  = C_BMG160_Zero_U8X;

	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {
			comres += p_bmg160->BMG160_BUS_WRITE_FUNC(reg,&v_data_u8r, 1);
	}
	return comres;
}

static int tmp_reg=0;

static ssize_t bmg_show_reg(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	size_t count = 0;
	u8 reg;

	BMG160_RETURN_FUNCTION_TYPE comres  = C_BMG160_Zero_U8X;

	if (p_bmg160 == BMG160_NULL) {
		return  E_BMG160_NULL_PTR;
	} else {

		comres = p_bmg160->BMG160_BUS_READ_FUNC(tmp_reg, &reg, 1);
		count += sprintf(&buf[count], "0x%x: %d\n", tmp_reg, reg);
	}

	return count;

}




static ssize_t bmg_store_reg(struct device *dev,
		struct device_attribute *attr,
		const char *buf, size_t count)
{
	int err;

	int address, value;


	sscanf(buf, "%3d %3d", &address, &value);


	if (value == 0xffff)
	{
		tmp_reg = address;
		return count;
	}


	err = BMG_CALL_API(store_reg)(address,value);


	if (err)
		return err;
	else
		return count;
}


static ssize_t bmg_set_power(struct device *dev,
		struct device_attribute *attr,
		const char *buf, size_t count)
{
	int enable;
	struct input_dev *input = to_input_dev(dev);
	struct bmg_client_data *client_data = input_get_drvdata(input);

	sscanf(buf, "%d", &enable);

	bmg160_power_ctrl(client_data->client , enable);

	return count;
}




static DEVICE_ATTR(chip_id, S_IRUGO,
		bmg_show_chip_id, NULL);
static DEVICE_ATTR(enable, S_IRUGO|S_IWUSR,
		bmg_show_enable, bmg_store_enable);

static DEVICE_ATTR(enable_by_read, S_IRUGO,
		bmg_enable_by_read, NULL);

static DEVICE_ATTR(disable_by_read, S_IRUGO,
		bmg_disable_by_read, NULL);

static DEVICE_ATTR(reg, S_IRUGO|S_IWUSR|S_IRGRP|S_IROTH,
		bmg_show_reg, bmg_store_reg);
static DEVICE_ATTR(spower, S_IWUSR|S_IRUGO,
		NULL, bmg_set_power);
static DEVICE_ATTR(fifo_framecount, S_IRUGO,
		bmg_show_fifo_framecount,NULL);
static DEVICE_ATTR(fifo_data_frame, S_IRUGO|S_IWUSR,
		bmg_show_fifo_data_frame, NULL);

static DEVICE_ATTR(value, S_IRUGO,
		bmg_show_value, NULL);


static struct attribute *bmg_attributes[] = {
	&dev_attr_chip_id.attr,
	&dev_attr_enable.attr,
	&dev_attr_enable_by_read.attr,
	&dev_attr_disable_by_read.attr,
	&dev_attr_reg.attr,
	&dev_attr_spower.attr,
	&dev_attr_fifo_framecount.attr,
	&dev_attr_fifo_data_frame.attr,
	&dev_attr_value.attr,
	NULL
};

static struct attribute_group bmg_attribute_group = {
	.attrs = bmg_attributes
};


static int bmg_input_init(struct bmg_client_data *client_data)
{
	struct input_dev *dev;
	int err = 0;

	dev = input_allocate_device();
	if (NULL == dev)
		return -ENOMEM;

	dev->name = SENSOR_NAME;
	dev->id.bustype = BUS_I2C;

	input_set_capability(dev, EV_ABS, ABS_MISC);
	input_set_abs_params(dev, ABS_X, BMG_VALUE_MIN, BMG_VALUE_MAX, 0, 0);
	input_set_abs_params(dev, ABS_Y, BMG_VALUE_MIN, BMG_VALUE_MAX, 0, 0);
	input_set_abs_params(dev, ABS_Z, BMG_VALUE_MIN, BMG_VALUE_MAX, 0, 0);
	input_set_drvdata(dev, client_data);

	err = input_register_device(dev);
	if (err < 0) {
		input_free_device(dev);
		return err;
	}
	client_data->input = dev;

	return 0;
}

static void bmg_input_destroy(struct bmg_client_data *client_data)
{
	struct input_dev *dev = client_data->input;

	input_unregister_device(dev);
	input_free_device(dev);
}

static int bmg_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int err = 0;
	struct bmg_client_data *client_data = NULL;
	struct regulator *v1p8_reg, *v2p8_reg;

	dev_info(&client->dev, "function entrance");

	v1p8_reg = devm_regulator_get(&client->dev, "V1P8SX");
	if (IS_ERR(v1p8_reg)) {
		dev_err(&client->dev, "Failure to retrieve V1P8SX regulator\n");
		return PTR_ERR(v1p8_reg);
	}

	v2p8_reg = devm_regulator_get(&client->dev, "V2P8SX");
	if (IS_ERR(v2p8_reg)) {
		dev_err(&client->dev, "Failure to retrieve V2P8SX regulator\n");
		return PTR_ERR(v2p8_reg);
	}

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		dev_err(&client->dev, "i2c_check_functionality error!");
		err = -EIO;
		goto exit_err_clean;
	}

	if (NULL == bmg_client) {
		bmg_client = client;
	} else {
		dev_err(&client->dev,
			"this driver does not support multiple clients");
		err = -EINVAL;
		goto exit_err_clean;
	}

	client_data = kzalloc(sizeof(struct bmg_client_data), GFP_KERNEL);
	if (NULL == client_data) {
		dev_err(&client->dev, "no memory available");
		err = -ENOMEM;
		goto exit_err_clean;
	}

	client_data->v1p8_reg = v1p8_reg;
	client_data->v2p8_reg = v2p8_reg;

	i2c_set_clientdata(client, client_data);
	client_data->client = client;

	bmg160_power_ctrl(client , 1);

	/* do soft reset */
	mdelay(5);
#if 0
	err = bmg_set_soft_reset(client);

	if (err < 0) {
		dev_err(&client->dev,
			"erro soft reset!\n");
		err = -EINVAL;
		goto exit_err_clean;
	}
#endif
	mdelay(30);

	/* check chip id */
	err = bmg_check_chip_id(client);
	if (!err) {
		dev_notice(&client->dev,
			"Bosch Sensortec Device %s detected", SENSOR_NAME);
	} else {
		dev_err(&client->dev,
			"Bosch Sensortec Device not found, chip id mismatch");
		err = -1;
		goto exit_err_clean;
	}

	/* input device init */
	err = bmg_input_init(client_data);
	if (err < 0)
		goto exit_err_clean;

	/* sysfs node creation */
	err = sysfs_create_group(&client_data->input->dev.kobj,
			&bmg_attribute_group);

	if (err < 0)
		goto exit_err_sysfs;


	/* h/w init */
	client_data->device.bus_read = bmg_i2c_read_wrapper;
	client_data->device.bus_write = bmg_i2c_write_wrapper;
	client_data->device.delay_msec = bmg_i2c_delay;
	BMG_CALL_API(init)(&client_data->device);


	dev_notice(&client->dev, "sensor %s probed successfully", SENSOR_NAME);

	dev_dbg(&client->dev,
		"i2c_client: %p client_data: %p i2c_device: %p input: %p",
		client, client_data, &client->dev, client_data->input);

	bmg160_power_ctrl(client , 0);
	return 0;

exit_err_sysfs:
	if (err)
		bmg_input_destroy(client_data);

exit_err_clean:
	if (err) {
		if (client_data != NULL) {
			kfree(client_data);
			client_data = NULL;
		}

		bmg_client = NULL;
	}

	bmg160_power_ctrl(client , 0);
	return err;
}


void bmg_shutdown(struct i2c_client *client)
{
	
}

static int bmg_remove(struct i2c_client *client)
{
	int err = 0;


	struct bmg_client_data *client_data =
		(struct bmg_client_data *)i2c_get_clientdata(client);

	

	if (NULL != client_data) {


		sysfs_remove_group(&client_data->input->dev.kobj,
				&bmg_attribute_group);
		bmg_input_destroy(client_data);
		kfree(client_data);

		bmg_client = NULL;
	}

	return err;
}





static const struct i2c_device_id bmg_id[] = {
	{ SENSOR_NAME, 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, bmg_id);


static struct acpi_device_id bmg_acpi_match[] = {
	{ "INT35CC" },
	{},
};

MODULE_DEVICE_TABLE(acpi, bmg_acpi_match);

static struct i2c_driver bmg_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = SENSOR_NAME,
		.acpi_match_table = ACPI_PTR(bmg_acpi_match),
	},
	.class = I2C_CLASS_HWMON,
	.id_table = bmg_id,
	.probe = bmg_probe,
	.remove = bmg_remove,
	.shutdown = bmg_shutdown,
};

static int __init BMG_init(void)
{
	return i2c_add_driver(&bmg_driver);
}

static void __exit BMG_exit(void)
{
	i2c_del_driver(&bmg_driver);
}

MODULE_AUTHOR("contact@bosch-sensortec.com>");
MODULE_DESCRIPTION("BMG GYROSCOPE SENSOR DRIVER");
MODULE_LICENSE("GPL v2");

module_init(BMG_init);
module_exit(BMG_exit);
