/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "mhs-error.h"
#include "mhs-error-private.h"

#include "mhs-history.h"
#include "mhs-history-bindings.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

static void mhs_history_dispose (GObject *object);

G_DEFINE_TYPE (MhsHistory, mhs_history, G_TYPE_OBJECT);

#define MHS_HISTORY_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_HISTORY, \
                                MhsHistoryPrivate))

typedef struct _MhsHistoryClosure MhsHistoryClosure;

struct _MhsHistoryPrivate
{
  DBusGProxy *proxy;

  guint callback_id;

  GSList *callbacks;
};

struct _MhsHistoryClosure
{
  MhsHistory *history;
  DBusGProxyCall *proxy_call;
  guint id;

  gpointer callback;
  gpointer user_data;
  GDestroyNotify user_data_notify;
};

enum
  {
    AC_RESULT_RECEIVED_SIGNAL,
    LINK_VISITED_SIGNAL,
    FAVORITES_RECEIVED_SIGNAL,
    PINNED_PAGE_SIGNAL,
    UNPINNED_PAGE_SIGNAL,

    LAST_SIGNAL
  };

static guint history_signals[LAST_SIGNAL] = { 0, };

static void
mhs_history_class_init (MhsHistoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->dispose = mhs_history_dispose;

  history_signals[AC_RESULT_RECEIVED_SIGNAL] =
    g_signal_new ("ac-result-received",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsHistoryClass, ac_result_received),
                  NULL, NULL,
                  mhs_marshal_VOID__UINT_STRING_STRING,
                  G_TYPE_NONE, 3,
                  G_TYPE_UINT,
                  G_TYPE_STRING,
                  G_TYPE_STRING);

  history_signals[LINK_VISITED_SIGNAL] =
    g_signal_new ("link-visited",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsHistoryClass, link_visited),
                  NULL, NULL,
                  mhs_marshal_VOID__STRING_INT,
                  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_INT);
  history_signals[FAVORITES_RECEIVED_SIGNAL] =
    g_signal_new ("favorites-received",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsHistoryClass, favorites_received),
                  NULL, NULL,
                  mhs_marshal_VOID__BOXED_BOXED,
                  G_TYPE_NONE, 2, G_TYPE_STRV, G_TYPE_STRV);

  history_signals[PINNED_PAGE_SIGNAL] =
    g_signal_new ("pinned-page",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsHistoryClass, pinned_page),
                  NULL, NULL,
                  mhs_marshal_VOID__STRING_STRING_INT,
                  G_TYPE_NONE, 3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT);

  history_signals[UNPINNED_PAGE_SIGNAL] =
    g_signal_new ("unpinned-page",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsHistoryClass, unpinned_page),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (MhsHistoryPrivate));

  dbus_g_object_register_marshaller (mhs_marshal_VOID__UINT_STRING_STRING,
                                     G_TYPE_NONE,
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
  dbus_g_object_register_marshaller (mhs_marshal_VOID__BOXED_BOXED,
                                     G_TYPE_NONE,
                                     G_TYPE_STRV,
                                     G_TYPE_STRV,
                                     G_TYPE_INVALID);
  dbus_g_object_register_marshaller (mhs_marshal_VOID__STRING_INT,
                                     G_TYPE_NONE,
                                     G_TYPE_STRING,
                                     G_TYPE_INT,
                                     G_TYPE_INVALID);
  dbus_g_object_register_marshaller (mhs_marshal_VOID__STRING_STRING_INT,
                                     G_TYPE_NONE,
                                     G_TYPE_STRING,
                                     G_TYPE_STRING,
                                     G_TYPE_INT,
                                     G_TYPE_INVALID);
}

static void
mhs_history_ac_result_received_cb (DBusGProxy *proxy,
                                      guint search_id,
                                      const gchar *value,
                                      const gchar *comment,
                                      gpointer data)
{
  g_signal_emit (data, history_signals[AC_RESULT_RECEIVED_SIGNAL], 0,
                 search_id, value, comment);
}

static void
mhs_history_link_visited_cb (DBusGProxy  *proxy,
                             const gchar *uri,
                             gint         visit_time,
                             gpointer     data)
{
  g_signal_emit (data, history_signals[LINK_VISITED_SIGNAL], 0,
                 uri, visit_time);
}

static void
mhs_history_favorites_received_cb (DBusGProxy  *proxy,
                                   gchar **urls,
                                   gchar **titles,
                                   gpointer data)
{
  g_signal_emit (data, history_signals[FAVORITES_RECEIVED_SIGNAL], 0,
                 urls, titles);
}

static void
mhs_history_pinned_page_cb (DBusGProxy  *proxy,
                            gchar *title,
                            gchar *url,
                            glong visit_time,
                            gpointer data)
{
  g_signal_emit (data, history_signals[PINNED_PAGE_SIGNAL], 0,
                 title, url, visit_time);
}

static void
mhs_history_unpinned_page_cb (DBusGProxy  *proxy,
                              gchar *url,
                              gpointer data)
{
  g_signal_emit (data, history_signals[UNPINNED_PAGE_SIGNAL], 0, url);
}

static void
mhs_history_init (MhsHistory *self)
{
  MhsHistoryPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = MHS_HISTORY_GET_PRIVATE (self);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy
        = dbus_g_proxy_new_for_name (connection,
                                     MHS_SERVICE_HISTORY,
                                     MHS_SERVICE_HISTORY_PATH,
                                     MHS_SERVICE_HISTORY_INTERFACE);

      dbus_g_connection_unref (connection);

      dbus_g_proxy_add_signal (priv->proxy, "AcResultReceived",
                               G_TYPE_UINT,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "LinkVisited",
                               G_TYPE_STRING,
                               G_TYPE_INT,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "FavoritesReceived",
                               G_TYPE_STRV,
                               G_TYPE_STRV,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "PinnedPage",
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INT,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "UnpinnedPage",
                               G_TYPE_STRING,
                               G_TYPE_INVALID);

      dbus_g_proxy_connect_signal
        (priv->proxy, "AcResultReceived",
         G_CALLBACK (mhs_history_ac_result_received_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "LinkVisited",
         G_CALLBACK (mhs_history_link_visited_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "FavoritesReceived",
         G_CALLBACK (mhs_history_favorites_received_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "PinnedPage",
         G_CALLBACK (mhs_history_pinned_page_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "UnpinnedPage",
         G_CALLBACK (mhs_history_unpinned_page_cb),
         self, NULL);
    }
}

static void
mhs_history_dispose (GObject *object)
{
  MhsHistory *self = (MhsHistory *) object;
  MhsHistoryPrivate *priv = self->priv;

  if (priv->proxy)
    {
      while (priv->callbacks)
        {
          MhsHistoryClosure *closure
            = priv->callbacks->data;
          /* This should also destroy the closure and remove it from
             the list */
          dbus_g_proxy_cancel_call (priv->proxy, closure->proxy_call);
        }

      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  G_OBJECT_CLASS (mhs_history_parent_class)->dispose (object);
}

MhsHistory *
mhs_history_new (void)
{
  MhsHistory *self = (MhsHistory *) g_object_new (MHS_TYPE_HISTORY,
                                                        NULL);

  return self;
}

static gboolean
mhs_history_check_proxy (MhsHistory *self,
                            GError **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, MHS_ERROR,
               MHS_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

gboolean
mhs_history_add_uri (MhsHistory *self,
                        const gchar *uri,
                        gboolean redirect,
                        gboolean top_level,
                        const gchar *referrer,
                        GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_add_uri (priv->proxy,
                                           uri,
                                           redirect,
                                           top_level,
                                           referrer,
                                           error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_history_set_page_title (MhsHistory *self,
                               const gchar *uri,
                               const gchar *title,
                               GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_set_page_title (priv->proxy,
                                                  uri, title,
                                                  error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_history_is_visited (MhsHistory *self,
                           const gchar *uri,
                           gboolean *is_visited,
                           GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_is_visited (priv->proxy,
                                              uri, is_visited,
                                              error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_history_start_ac_search (MhsHistory *self,
                                const gchar *search_str,
                                guint32 *search_id,
                                GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_start_ac_search (priv->proxy,
                                                   search_str, search_id,
                                                   error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_history_stop_ac_search (MhsHistory *self,
                               guint32 search_id,
                               GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_stop_ac_search (priv->proxy,
                                                  search_id,
                                                  error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_history_set_favicon_url (MhsHistory *self,
                                const gchar *page_uri,
                                const gchar *favicon_uri,
                                GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_set_favicon_url (priv->proxy,
                                                   page_uri, favicon_uri,
                                                   error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_history_set_default_favicon_url (MhsHistory *self,
                                        const gchar *page_uri,
                                        GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_set_default_favicon_url (priv->proxy,
                                                           page_uri,
                                                           error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

static void
mhs_history_closure_destroy_cb (gpointer data)
{
  MhsHistoryClosure *closure = data;
  MhsHistoryPrivate *priv = closure->history->priv;

  if (closure->user_data_notify)
    closure->user_data_notify (closure->user_data);

  priv->callbacks = g_slist_remove (priv->callbacks, closure);

  g_slice_free (MhsHistoryClosure, closure);
}

static void
mhs_history_get_favicon_cb (DBusGProxy *proxy,
                               DBusGProxyCall *proxy_call,
                               void *data)
{
  MhsHistoryClosure *closure = data;
  GError *error = NULL;
  gchar *mime_type = NULL;
  GArray *icon_data = NULL;

  if (dbus_g_proxy_end_call (proxy, proxy_call, &error,
                             G_TYPE_STRING, &mime_type,
                             DBUS_TYPE_G_UCHAR_ARRAY, &icon_data,
                             G_TYPE_INVALID))
    {
      ((MhsHistoryFaviconCallback) closure->callback)
        (closure->history,
         mime_type,
         (guint8 *) icon_data->data,
         icon_data->len,
         NULL,
         closure->user_data);

      g_free (mime_type);
      g_array_free (icon_data, TRUE);
    }
  else
    {
      _mhs_error_translate_from_dbus (&error);

      ((MhsHistoryFaviconCallback) closure->callback)
        (closure->history,
         NULL,
         NULL, 0,
         error,
         closure->user_data);

      g_error_free (error);
    }
}

guint
mhs_history_get_favicon (MhsHistory *self,
                            const gchar *page_uri,
                            gboolean download,
                            MhsHistoryFaviconCallback callback,
                            gpointer user_data,
                            GDestroyNotify user_data_notify)
{
  MhsHistoryPrivate *priv;
  MhsHistoryClosure *closure;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);

  priv = self->priv;

  if (++priv->callback_id == 0)
    priv->callback_id = 1;

  closure = g_slice_new0 (MhsHistoryClosure);
  closure->history = self;
  closure->id = priv->callback_id;
  closure->callback = callback;
  closure->user_data = user_data;
  closure->user_data_notify = user_data_notify;

  closure->proxy_call
    = dbus_g_proxy_begin_call (priv->proxy,
                               "GetFavicon",
                               mhs_history_get_favicon_cb,
                               closure,
                               mhs_history_closure_destroy_cb,
                               G_TYPE_STRING,
                               page_uri,
                               G_TYPE_BOOLEAN,
                               download,
                               G_TYPE_INVALID);

  priv->callbacks = g_slist_prepend (priv->callbacks, closure);

  return closure->id;
}

static void
mhs_history_get_favorites_cb (DBusGProxy *proxy,
                              GError *error,
                              gpointer userdata)
{
  /* We don't need to do anything. The service will fire a signal when
     it actually gets the results */
}

void
mhs_history_get_favorites (MhsHistory *self)
{
  MhsHistoryPrivate *priv;

  g_return_if_fail (MHS_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mhs_History_get_favorites_async (priv->proxy,
                                              mhs_history_get_favorites_cb,
                                              self);
}

gboolean
mhs_history_clear_history (MhsHistory *self,
                           GError **error)
{
  MhsHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_History_clear_history (priv->proxy,
                                              error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

static void
mhs_history_get_is_page_pinned_cb (DBusGProxy *proxy,
                                   DBusGProxyCall *proxy_call,
                                   void *data)
{
  MhsHistoryClosure *closure = data;
  GError *error = NULL;
  gboolean is_page_pinned;

  if (dbus_g_proxy_end_call (proxy, proxy_call, &error,
                             G_TYPE_BOOLEAN, &is_page_pinned,
                             G_TYPE_INVALID))
    ((MhsHistoryIsPagePinnedCallback) closure->callback)
      (closure->history,
       is_page_pinned,
       NULL,
       closure->user_data);
  else
    {
      _mhs_error_translate_from_dbus (&error);

      ((MhsHistoryIsPagePinnedCallback) closure->callback)
        (closure->history,
         FALSE,
         error,
         closure->user_data);

      g_error_free (error);
    }
}

guint
mhs_history_get_is_page_pinned (MhsHistory *self,
                                const gchar *page_uri,
                                MhsHistoryIsPagePinnedCallback callback,
                                gpointer user_data,
                                GDestroyNotify user_data_notify)
{
  MhsHistoryPrivate *priv;
  MhsHistoryClosure *closure;

  g_return_val_if_fail (MHS_IS_HISTORY (self), FALSE);

  priv = self->priv;

  if (++priv->callback_id == 0)
    priv->callback_id = 1;

  closure = g_slice_new0 (MhsHistoryClosure);
  closure->history = self;
  closure->id = priv->callback_id;
  closure->callback = callback;
  closure->user_data = user_data;
  closure->user_data_notify = user_data_notify;

  closure->proxy_call
    = dbus_g_proxy_begin_call (priv->proxy,
                               "GetIsPagePinned",
                               mhs_history_get_is_page_pinned_cb,
                               closure,
                               mhs_history_closure_destroy_cb,
                               G_TYPE_STRING,
                               page_uri,
                               G_TYPE_INVALID);

  priv->callbacks = g_slist_prepend (priv->callbacks, closure);

  return closure->id;
}

static void
mhs_history_get_pinned_pages_cb (DBusGProxy *proxy,
                                 GError *error,
                                 gpointer userdata)
{
}

void
mhs_history_get_pinned_pages (MhsHistory *self)
{
  MhsHistoryPrivate *priv;

  g_return_if_fail (MHS_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mhs_History_get_pinned_pages_async
    (priv->proxy,
     mhs_history_get_pinned_pages_cb,
     self);
}

static void
mhs_history_pin_page_cb (DBusGProxy *proxy,
                         GError *error,
                         gpointer userdata)
{
}

void
mhs_history_pin_page (MhsHistory *self,
                      const gchar *uri,
                      const gchar *title)
{
  MhsHistoryPrivate *priv;

  g_return_if_fail (MHS_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mhs_History_pin_page_async (priv->proxy,
                                         uri, title,
                                         mhs_history_pin_page_cb,
                                         self);
}

static void
mhs_history_unpin_page_cb (DBusGProxy *proxy,
                           GError *error,
                           gpointer userdata)
{
}

void
mhs_history_unpin_page (MhsHistory *self,
                        const gchar *uri)
{
  MhsHistoryPrivate *priv;

  g_return_if_fail (MHS_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mhs_History_unpin_page_async (priv->proxy,
                                           uri,
                                           mhs_history_unpin_page_cb,
                                           self);
}

static void
mhs_history_unpin_all_pages_cb (DBusGProxy *proxy,
                                GError *error,
                                gpointer userdata)
{
}

void
mhs_history_unpin_all_pages (MhsHistory *self)
{
  MhsHistoryPrivate *priv;

  g_return_if_fail (MHS_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mhs_History_unpin_all_pages_async (priv->proxy,
                                                mhs_history_unpin_all_pages_cb,
                                                self);
}

void
mhs_history_cancel (MhsHistory *self,
                       guint id)
{
  MhsHistoryPrivate *priv;
  MhsHistoryClosure *closure;
  GSList *l;

  g_return_if_fail (MHS_IS_HISTORY (self));

  priv = self->priv;

  for (l = priv->callbacks; l; l = l->next)
    {
      closure = l->data;

      if (closure->id == id)
        {
          dbus_g_proxy_cancel_call (priv->proxy,
                                    closure->proxy_call);
          break;
        }
    }
}
