/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <nsIComponentManager.h>
#include <nsILoginManagerStorage.h>
#include <nsILoginManager.h>
#include <nsILoginInfo.h>
#include <nsIProperty.h>
#include <nsIPropertyBag.h>
#include <nsServiceManagerUtils.h>
#include <nsComponentManagerUtils.h>
#include <nsStringAPI.h>
#include <nsIArray.h>
#include <nsIMutableArray.h>
#include <nsCOMPtr.h>
#include <nsMemory.h>
#include <nsIVariant.h>
#include <new>

#include "mhs-login-manager-storage.h"
#include "mhs-error-private.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

/* D-Bus method implementations */

static gboolean mhs_lms_add_login (MhsLoginManagerStorage *lms,
                                   const gchar *hostname,
                                   const gchar *form_submit_url,
                                   const gchar *http_realm,
                                   const gchar *username,
                                   const gchar *password,
                                   const gchar *username_field,
                                   const gchar *password_field,
                                   guint32 null_fields,
                                   GError **error);

static gboolean mhs_lms_count_logins (MhsLoginManagerStorage *lms,
                                      const gchar *hostname,
                                      const gchar *action_url,
                                      const gchar *http_realm,
                                      guint32 null_fields,
                                      unsigned long *count_p,
                                      GError **error);

static gboolean mhs_lms_find_logins (MhsLoginManagerStorage *lms,
                                     const gchar *hostname,
                                     const gchar *action_url,
                                     const gchar *http_realm,
                                     guint32 null_fields,
                                     GPtrArray **logins,
                                     GError **error);

static gboolean mhs_lms_get_all_disabled_hosts (MhsLoginManagerStorage *lms,
                                                gchar ***hostnames_out,
                                                GError **error);

static gboolean mhs_lms_get_all_logins (MhsLoginManagerStorage *lms,
                                        GPtrArray **logins,
                                        GError **error);

static gboolean mhs_lms_get_all_encrypted_logins (MhsLoginManagerStorage *lms,
                                                  GPtrArray **logins,
                                                  GError **error);

static gboolean mhs_lms_get_login_saving_enabled (MhsLoginManagerStorage *lms,
                                                  const gchar *host,
                                                  gboolean *enabled_out,
                                                  GError **error);

static gboolean mhs_lms_init (MhsLoginManagerStorage *lms,
                              GError **error);

static gboolean mhs_lms_init_with_file (MhsLoginManagerStorage *lms,
                                        const gchar *input_file,
                                        const gchar *output_file,
                                        GError **error);

static gboolean mhs_lms_modify_login (MhsLoginManagerStorage *lms,
                                      GValueArray *old_login,
                                      GHashTable *new_values,
                                      GError **error);

static gboolean mhs_lms_remove_all_logins (MhsLoginManagerStorage *lms,
                                           GError **error);

static gboolean mhs_lms_remove_login (MhsLoginManagerStorage *lms,
                                      GValueArray *old_login,
                                      GError **error);

static gboolean mhs_lms_search_logins (MhsLoginManagerStorage *lms,
                                       GHashTable *match_data,
                                       GPtrArray **logins,
                                       GError **error);

static gboolean mhs_lms_set_login_saving_enabled (MhsLoginManagerStorage *lms,
                                                  const gchar *host,
                                                  gboolean is_enabled,
                                                  GError **error);

#include "mhs-login-manager-storage-glue.h"

/* End D-Bus method implementations */

/* A simple nsIPropertyBag implementation which wraps a string array */
class MhsLmsPropertyBag : public nsIPropertyBag
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIPROPERTYBAG

  MhsLmsPropertyBag (GHashTable *values);

private:
  nsCOMPtr<nsIArray> properties;
};

class MhsLmsProperty : public nsIProperty
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIPROPERTY

  MhsLmsProperty (const gchar *key_in, const gchar *value_in);

private:
  NS_ConvertUTF8toUTF16 key;
  nsCOMPtr<nsIVariant> value;
};

/* Helper class to convert a C-style UTF-8 string to an nsString or to
   set it to void if the pointer is NULL */
class MhsConvertStringOrNull : public nsString
{
public:
  MhsConvertStringOrNull (const gchar *str)
  {
    if (str)
      NS_CStringToUTF16 (nsDependentCString (str),
                         NS_CSTRING_ENCODING_UTF8,
                         *this);
    else
      SetIsVoid (true);
  }
};

static void mhs_login_manager_storage_finalize (GObject *object);

G_DEFINE_TYPE (MhsLoginManagerStorage,
               mhs_login_manager_storage,
               G_TYPE_OBJECT);

#define MHS_LOGIN_MANAGER_STORAGE_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_LOGIN_MANAGER_STORAGE, \
                                MhsLoginManagerStoragePrivate))

struct _MhsLoginManagerStoragePrivate
{
  nsCOMPtr<nsILoginManagerStorage> lms_service;
};

static void
mhs_login_manager_storage_class_init (MhsLoginManagerStorageClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = mhs_login_manager_storage_finalize;

  g_type_class_add_private (klass, sizeof (MhsLoginManagerStoragePrivate));

  dbus_g_object_type_install_info
    (MHS_TYPE_LOGIN_MANAGER_STORAGE,
     &dbus_glib_mhs_lms_object_info);
}

static void
mhs_login_manager_storage_init (MhsLoginManagerStorage *self)
{
  MhsLoginManagerStoragePrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;
  nsresult rv;

  priv = self->priv = MHS_LOGIN_MANAGER_STORAGE_GET_PRIVATE (self);

  /* 'placement new' to call the constructor for
     MhsLoginManagerStoragePrivate */
  new (reinterpret_cast<void *> (priv)) MhsLoginManagerStoragePrivate;

  priv->lms_service
    = do_GetService ("@mozilla.org/login-manager/storage/mozStorage;1", &rv);
  if (NS_FAILED (rv))
    g_warning ("Failed to retrieve login manager service");

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object
        (connection,
         MHS_SERVICE_LOGIN_MANAGER_STORAGE_PATH,
         G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }
}

static void
mhs_login_manager_storage_finalize (GObject *object)
{
  MhsLoginManagerStorage *self = MHS_LOGIN_MANAGER_STORAGE (object);
  MhsLoginManagerStoragePrivate *priv = self->priv;

  /* Explicitly call the destructor for the private data (so that it
     destructs without trying to free the memory) */
  priv->~MhsLoginManagerStoragePrivate ();

  G_OBJECT_CLASS (mhs_login_manager_storage_parent_class)->finalize (object);
}

MhsLoginManagerStorage *
mhs_login_manager_storage_new (void)
{
  MhsLoginManagerStorage *self
    = (MhsLoginManagerStorage *) g_object_new (MHS_TYPE_LOGIN_MANAGER_STORAGE,
                                               NULL);
  return self;
}

static gboolean
mhs_lms_add_login (MhsLoginManagerStorage *lms,
                   const gchar *hostname,
                   const gchar *form_submit_url,
                   const gchar *http_realm,
                   const gchar *username,
                   const gchar *password,
                   const gchar *username_field,
                   const gchar *password_field,
                   guint32 null_fields,
                   GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;

  /* Create an nsILoginInfo instance to contain the parameters */
  nsCOMPtr<nsILoginInfo> login_info
    = do_CreateInstance ("@mozilla.org/login-manager/loginInfo;1", &rv);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = login_info->Init (MhsConvertStringOrNull ((null_fields & 1)
                                                 ? NULL : hostname),
                         MhsConvertStringOrNull ((null_fields & 2)
                                                 ? NULL : form_submit_url),
                         MhsConvertStringOrNull ((null_fields & 4)
                                                 ? NULL : http_realm),
                         MhsConvertStringOrNull ((null_fields & 8)
                                                 ? NULL : username),
                         MhsConvertStringOrNull ((null_fields & 16)
                                                 ? NULL : password),
                         MhsConvertStringOrNull ((null_fields & 32)
                                                 ? NULL : username_field),
                         MhsConvertStringOrNull ((null_fields & 64)
                                                 ? NULL : password_field));
  if (NS_FAILED (rv))
    goto rv_error;

  rv = priv->lms_service->AddLogin (login_info);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_count_logins (MhsLoginManagerStorage *lms,
                      const gchar *hostname,
                      const gchar *action_url,
                      const gchar *http_realm,
                      guint32 null_fields,
                      unsigned long *count_p,
                      GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRUint32 count;

  rv = priv->lms_service->CountLogins
    (MhsConvertStringOrNull ((null_fields & 1) ? NULL : hostname),
     MhsConvertStringOrNull ((null_fields & 2) ? NULL : action_url),
     MhsConvertStringOrNull ((null_fields & 4) ? NULL : http_realm),
                             &count);
  if (NS_FAILED (rv))
    goto rv_error;

  *count_p = count;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

nsresult typedef (nsILoginInfo::*MhsLmsLoginInfoGetter) (nsAString &);

static MhsLmsLoginInfoGetter mhs_lms_login_info_getters[] =
  {
    &nsILoginInfo::GetHostname,
    &nsILoginInfo::GetFormSubmitURL,
    &nsILoginInfo::GetHttpRealm,
    &nsILoginInfo::GetUsername,
    &nsILoginInfo::GetPassword,
    &nsILoginInfo::GetUsernameField,
    &nsILoginInfo::GetPasswordField
  };

static gboolean
mhs_lms_logins_to_gptrarray (PRUint32 count,
                             nsILoginInfo **login_infos,
                             GPtrArray **logins,
                             GError **error)
{
  PRUint32 login_num, getter_num;
  gboolean ret = TRUE;
  GValue empty_string_value = { 0 };

  g_value_init (&empty_string_value, G_TYPE_STRING);

  *logins = g_ptr_array_new ();

  /* Convert each login to a GValueArray */
  for (login_num = 0; login_num < count; login_num++)
    {
      GValueArray *value_array
        = g_value_array_new (G_N_ELEMENTS (mhs_lms_login_info_getters));
      guint32 null_fields = 0;
      GValue null_fields_value = { 0 };

      g_ptr_array_add (*logins, value_array);

      for (getter_num = 0;
           getter_num < G_N_ELEMENTS (mhs_lms_login_info_getters);
           getter_num++)
        {
          nsresult rv;
          nsAutoString str_value;
          GValue *value;

          rv = (login_infos[login_num]
                ->*mhs_lms_login_info_getters[getter_num]) (str_value);
          if (NS_FAILED (rv))
            {
              mhs_error_set_from_nsresult (rv, error);
              ret = FALSE;
              goto skip;
            }

          g_value_array_append (value_array, &empty_string_value);
          if (str_value.IsVoid ())
            null_fields |= (1 << getter_num);
          else
            {
              value = g_value_array_get_nth (value_array, getter_num);
              NS_ConvertUTF16toUTF8 value_utf8 (str_value);
              g_value_set_string (value, value_utf8.get ());
            }
        }

      /* The last value in the GValue array is an integer mask of the
         null fields */
      g_value_init (&null_fields_value, G_TYPE_UINT);
      g_value_set_uint (&null_fields_value, null_fields);
      g_value_array_append (value_array, &null_fields_value);
    }
 skip:

  for (login_num = 0; login_num < count; login_num++)
    login_infos[login_num]->Release ();

  nsMemory::Free (login_infos);

  if (!ret)
    {
      g_ptr_array_foreach (*logins, (GFunc) g_value_array_free, NULL);
      g_ptr_array_free (*logins, TRUE);
    }

  g_value_unset (&empty_string_value);

  return ret;
}

static gboolean
mhs_lms_find_logins (MhsLoginManagerStorage *lms,
                     const gchar *hostname,
                     const gchar *action_url,
                     const gchar *http_realm,
                     guint32 null_fields,
                     GPtrArray **logins,
                     GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRUint32 count;
  nsILoginInfo **login_infos;

  rv = priv->lms_service->FindLogins
    (&count,
     MhsConvertStringOrNull ((null_fields & 1) ? NULL : hostname),
     MhsConvertStringOrNull ((null_fields & 2) ? NULL : action_url),
     MhsConvertStringOrNull ((null_fields & 4) ? NULL : http_realm),
     &login_infos);
  if (NS_FAILED (rv))
    goto rv_error;

  return mhs_lms_logins_to_gptrarray (count, login_infos, logins, error);

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_get_all_disabled_hosts (MhsLoginManagerStorage *lms,
                                gchar ***hostnames_out,
                                GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRUint32 i, count;
  PRUnichar **hostnames;

  rv = priv->lms_service->GetAllDisabledHosts (&count, &hostnames);
  if (NS_FAILED (rv))
    goto rv_error;

  *hostnames_out = g_new (gchar *, count + 1);

  for (i = 0; i < count; i++)
    {
      NS_ConvertUTF16toUTF8 hostname_utf8 (hostnames[i]);
      (* hostnames_out)[i] = g_new (gchar, hostname_utf8.Length () + 1);
      memcpy ((* hostnames_out)[i], hostname_utf8.get (),
              hostname_utf8.Length ());
      (* hostnames_out)[i][hostname_utf8.Length ()] = '\0';
    }

  (* hostnames_out)[count] = NULL;

  for (i = 0; i < count; i++)
    nsMemory::Free (hostnames[i]);
  nsMemory::Free(hostnames);

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_get_all_logins (MhsLoginManagerStorage *lms,
                        GPtrArray **logins,
                        GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRUint32 count;
  nsILoginInfo **login_infos;

  rv = priv->lms_service->GetAllLogins (&count, &login_infos);
  if (NS_FAILED (rv))
    goto rv_error;

  return mhs_lms_logins_to_gptrarray (count, login_infos, logins, error);

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_get_all_encrypted_logins (MhsLoginManagerStorage *lms,
                                  GPtrArray **logins,
                                  GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRUint32 count;
  nsILoginInfo **login_infos;

  rv = priv->lms_service->GetAllEncryptedLogins (&count, &login_infos);
  if (NS_FAILED (rv))
    goto rv_error;

  return mhs_lms_logins_to_gptrarray (count, login_infos, logins, error);

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_get_login_saving_enabled (MhsLoginManagerStorage *lms,
                                  const gchar *host,
                                  gboolean *enabled_out,
                                  GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRBool enabled;

  rv = priv->lms_service->GetLoginSavingEnabled (NS_ConvertUTF8toUTF16 (host),
                                                 &enabled);
  if (NS_FAILED (rv))
    goto rv_error;

  *enabled_out = enabled;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_init (MhsLoginManagerStorage *lms,
              GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;

  rv = priv->lms_service->Init ();
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_init_with_file (MhsLoginManagerStorage *lms,
                        const gchar *input_file,
                        const gchar *output_file,
                        GError **error)
{
  mhs_error_set_from_nsresult (NS_ERROR_NOT_IMPLEMENTED, error);
  return TRUE;
}

nsresult typedef (nsILoginInfo::*MhsLmsLoginInfoSetter) (const nsAString &);

static MhsLmsLoginInfoSetter mhs_lms_login_info_setters[] =
  {
    &nsILoginInfo::SetHostname,
    &nsILoginInfo::SetFormSubmitURL,
    &nsILoginInfo::SetHttpRealm,
    &nsILoginInfo::SetUsername,
    &nsILoginInfo::SetPassword,
    &nsILoginInfo::SetUsernameField,
    &nsILoginInfo::SetPasswordField
  };

static nsresult
mhs_lms_convert_value_array_to_login_info (GValueArray *value_array,
                                           nsILoginInfo **login_info_out)
{
  nsresult rv;
  guint i;
  GValue *null_fields_value;
  guint null_fields;

  /* Convert the three parameters to an nsILoginInfo. The hope is that
     these are all that is used to identify the login */
  nsCOMPtr<nsILoginInfo> login_info
    = do_CreateInstance ("@mozilla.org/login-manager/loginInfo;1", &rv);
  if (NS_FAILED (rv))
    return rv;

  null_fields_value
    = g_value_array_get_nth (value_array,
                             G_N_ELEMENTS (mhs_lms_login_info_setters));
  null_fields = g_value_get_uint (null_fields_value);

  for (i = 0;
       i < MIN (value_array->n_values,
                G_N_ELEMENTS (mhs_lms_login_info_setters));
       i++)
    if ((null_fields & (1 << i)) == 0)
      {
        GValue *value = g_value_array_get_nth (value_array, i);

        if (g_value_get_string (value))
          (login_info->*mhs_lms_login_info_setters[i])
            (NS_ConvertUTF8toUTF16 (g_value_get_string (value)));
      }

  *login_info_out = login_info;
  (* login_info_out)->AddRef ();

  return NS_OK;
}

static gboolean
mhs_lms_modify_login (MhsLoginManagerStorage *lms,
                      GValueArray *old_login,
                      GHashTable *new_values,
                      GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;

  nsCOMPtr<nsIPropertyBag> bag;

  nsCOMPtr<nsILoginInfo> login_info;
  rv = mhs_lms_convert_value_array_to_login_info (old_login,
                                                  getter_AddRefs (login_info));
  if (NS_FAILED (rv))
    goto rv_error;

  /* Convert the new values into an nsIPropertyBag */
  bag = new MhsLmsPropertyBag (new_values);

  rv = priv->lms_service->ModifyLogin (login_info, bag);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_remove_all_logins (MhsLoginManagerStorage *lms,
                           GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;

  rv = priv->lms_service->RemoveAllLogins ();
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_remove_login (MhsLoginManagerStorage *lms,
                      GValueArray *old_login,
                      GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;

  nsCOMPtr<nsILoginInfo> login_info;
  rv = mhs_lms_convert_value_array_to_login_info (old_login,
                                                  getter_AddRefs (login_info));
  if (NS_FAILED (rv))
    goto rv_error;

  rv = priv->lms_service->RemoveLogin (login_info);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_search_logins (MhsLoginManagerStorage *lms,
                       GHashTable *match_data,
                       GPtrArray **logins,
                       GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;
  PRUint32 count;
  nsILoginInfo **login_infos;

  /* Convert the match data into an nsIPropertyBag */
  nsCOMPtr<nsIPropertyBag> bag = new MhsLmsPropertyBag (match_data);

  rv = priv->lms_service->SearchLogins (&count, bag, &login_infos);
  if (NS_FAILED (rv))
    goto rv_error;

  return mhs_lms_logins_to_gptrarray (count, login_infos, logins, error);

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_lms_set_login_saving_enabled (MhsLoginManagerStorage *lms,
                                  const gchar *host,
                                  gboolean is_enabled,
                                  GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_LOGIN_MANAGER_STORAGE (lms), FALSE);

  MhsLoginManagerStoragePrivate *priv = lms->priv;

  rv = priv->lms_service->SetLoginSavingEnabled (NS_ConvertUTF8toUTF16 (host),
                                                 is_enabled);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

NS_IMPL_ISUPPORTS1(MhsLmsPropertyBag, nsIPropertyBag)

static void
mhs_lms_add_hash_table_entry (gpointer key, gpointer value, gpointer user_data)
{
  nsIMutableArray *properties = static_cast<nsIMutableArray *> (user_data);

  properties->AppendElement (new MhsLmsProperty ((gchar *) key,
                                                 (gchar *) value),
                             PR_FALSE);
}

MhsLmsPropertyBag::MhsLmsPropertyBag (GHashTable *values)
: properties (do_CreateInstance (NS_ARRAY_CONTRACTID))
{
  nsCOMPtr<nsIMutableArray> mutable_properties
    = do_QueryInterface (properties);

  /* Convert the hash table to an array of properties */
  g_hash_table_foreach (values, mhs_lms_add_hash_table_entry,
                        mutable_properties);
}

/* readonly attribute nsISimpleEnumerator enumerator; */
NS_IMETHODIMP
MhsLmsPropertyBag::GetEnumerator (nsISimpleEnumerator **aEnumerator)
{
  return properties->Enumerate (aEnumerator);
}

/* nsIVariant getProperty (in AString name); */
NS_IMETHODIMP
MhsLmsPropertyBag::GetProperty (const nsAString &name,
                                nsIVariant **retval NS_OUTPARAM)
{
  PRUint32 length;
  nsAutoString prop_name;

  properties->GetLength (&length);

  for (PRUint32 i = 0; i < length; i++)
    {
      nsCOMPtr<nsIProperty> prop;

      properties->QueryElementAt (i, NS_GET_IID (nsIProperty),
                                  getter_AddRefs (prop));
      prop->GetName (prop_name);

      if (prop_name.Equals (name))
        return prop->GetValue (retval);
    }

  return NS_ERROR_NOT_AVAILABLE;
}

NS_IMPL_ISUPPORTS1(MhsLmsProperty, nsIProperty)

MhsLmsProperty::MhsLmsProperty (const gchar *key_in,
                                const gchar *value_in)
: key (key_in),
  value (do_CreateInstance (NS_VARIANT_CONTRACTID))
{
  nsCOMPtr<nsIWritableVariant> writable_value (do_QueryInterface (value));
  writable_value->SetAsAString (NS_ConvertUTF8toUTF16 (value_in));
}

/* readonly attribute AString name; */
NS_IMETHODIMP
MhsLmsProperty::GetName (nsAString &aName)
{
  aName.Assign (key);

  return NS_OK;
}

/* readonly attribute nsIVariant value; */
NS_IMETHODIMP
MhsLmsProperty::GetValue (nsIVariant **aValue)
{
  *aValue = value;
  NS_ADDREF (*aValue);

  return NS_OK;
}
