/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <new>
#include <nsILocalFile.h>
#include <nsIPrefService.h>
#include <nsIPrefBranch2.h>
#include <nsIIOService.h>
#include <nsIObserver.h>
#include <nsToolkitCompsCID.h>
#include <nsNetCID.h>
#include <nsServiceManagerUtils.h>
#include <nsCOMPtr.h>
#include <nsStringAPI.h>
#include <nsMemory.h>

#include "mhs-prefs.h"
#include "mhs-error-private.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

/* D-Bus method implementations */

static gboolean mhs_prefs_read_user (MhsPrefs  *self,
                                     const gchar  *path,
                                     GError      **error);
static gboolean mhs_prefs_reset (MhsPrefs  *self,
                                 GError      **error);
static gboolean mhs_prefs_reset_user (MhsPrefs  *self,
                                      GError      **error);
static gboolean mhs_prefs_save_pref_file (MhsPrefs  *self,
                                          const gchar  *path,
                                          GError      **error);
static gboolean mhs_prefs_get_branch (MhsPrefs  *self,
                                      const gchar  *root,
                                      gint         *id,
                                      GError      **error);
static gboolean mhs_prefs_get_default_branch (MhsPrefs  *self,
                                              const gchar  *root,
                                              gint         *id,
                                              GError      **error);
static gboolean mhs_prefs_release_branch (MhsPrefs  *self,
                                          gint          id,
                                          GError      **error);

static gboolean mhs_prefs_branch_get_type (MhsPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gint         *type,
                                           GError      **error);

static gboolean mhs_prefs_branch_get_bool (MhsPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gboolean     *value,
                                           GError      **error);

static gboolean mhs_prefs_branch_set_bool (MhsPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gboolean      value,
                                           GError      **error);

static gboolean mhs_prefs_branch_get_int (MhsPrefs  *self,
                                          gint          id,
                                          const gchar  *name,
                                          gint         *value,
                                          GError      **error);

static gboolean mhs_prefs_branch_set_int (MhsPrefs  *self,
                                          gint          id,
                                          const gchar  *name,
                                          gint          value,
                                          GError      **error);

static gboolean mhs_prefs_branch_get_char (MhsPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gchar       **value,
                                           GError      **error);

static gboolean mhs_prefs_branch_set_char (MhsPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           const gchar  *value,
                                           GError      **error);

static gboolean mhs_prefs_branch_has_user_value (MhsPrefs  *self,
                                                 gint          id,
                                                 const gchar  *name,
                                                 gboolean     *result,
                                                 GError      **error);

static gboolean mhs_prefs_branch_lock (MhsPrefs  *self,
                                       gint          id,
                                       const gchar  *name,
                                       GError      **error);

static gboolean mhs_prefs_branch_is_locked (MhsPrefs  *self,
                                            gint          id,
                                            const gchar  *name,
                                            gboolean     *result,
                                            GError      **error);

static gboolean mhs_prefs_branch_unlock (MhsPrefs  *self,
                                         gint          id,
                                         const gchar  *name,
                                         GError      **error);

static gboolean mhs_prefs_branch_get_child_list (MhsPrefs   *self,
                                                 gint           id,
                                                 const gchar   *start,
                                                 guint         *len,
                                                 gchar       ***array,
                                                 GError       **error);

static gboolean mhs_prefs_branch_add_observer (MhsPrefs  *self,
                                               gint          id,
                                               const gchar  *domain,
                                               GError      **error);

static gboolean mhs_prefs_branch_remove_observer (MhsPrefs  *self,
                                                  gint          id,
                                                  const gchar  *domain,
                                                  GError      **error);

#include "mhs-prefs-glue.h"

/* End D-Bus method implementations */

static void mhs_prefs_finalize (GObject *object);

G_DEFINE_TYPE (MhsPrefs, mhs_prefs, G_TYPE_OBJECT);

#define MHS_PREFS_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_PREFS, \
                                MhsPrefsPrivate))

class MhsPrefsObserver : public nsIObserver
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIOBSERVER

  MhsPrefs *mPrefs;
};

struct _MhsPrefsPrivate
{
  nsIPrefService   *pref_service;
  GHashTable       *branches_by_id;
  GHashTable       *ids_by_branch;
  MhsPrefsObserver  observer;
};

enum
  {
    READ_SIGNAL,
    RESET_SIGNAL,
    APP_DEFAULTS_SIGNAL,
    BRANCH_CHANGED_SIGNAL,

    LAST_SIGNAL
  };

static guint prefs_signals[LAST_SIGNAL] = { 0, };

NS_IMPL_QUERY_INTERFACE1(MhsPrefsObserver, nsIObserver)

/* We don't want to implement reference counting for
   MhsPrefsObserver because it is entirely owned by the
   MhsPrefs instance */
NS_IMETHODIMP_(nsrefcnt)
MhsPrefsObserver::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MhsPrefsObserver::Release ()
{
  return 1;
}

NS_IMETHODIMP
MhsPrefsObserver::Observe(nsISupports     *aSubject,
                          const char      *aTopic,
                          const PRUnichar *someData)
{
  MhsPrefsPrivate *priv = mPrefs->priv;
  nsIPrefBranch *branch = static_cast<nsIPrefBranch *>(aSubject);
  gint id = GPOINTER_TO_INT (
    g_hash_table_lookup (priv->ids_by_branch, branch));
  NS_ConvertUTF16toUTF8 domain (someData);
  g_signal_emit (mPrefs, prefs_signals[BRANCH_CHANGED_SIGNAL], 0,
                 id, domain.get ());
  return NS_OK;
}

static void
mhs_prefs_class_init (MhsPrefsClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = mhs_prefs_finalize;

  prefs_signals[READ_SIGNAL] =
    g_signal_new ("read",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, read),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[RESET_SIGNAL] =
    g_signal_new ("reset",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, reset),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[APP_DEFAULTS_SIGNAL] =
    g_signal_new ("app-defaults",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, app_defaults),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[BRANCH_CHANGED_SIGNAL] =
    g_signal_new ("branch-changed",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, branch_changed),
                  NULL, NULL,
                  mhs_marshal_VOID__INT_STRING,
                  G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (MhsPrefsPrivate));

  dbus_g_object_type_install_info (MHS_TYPE_PREFS,
                                   &dbus_glib_mhs_prefs_object_info);
}

static void
mhs_prefs_init (MhsPrefs *self)
{
  nsresult rv;
  MhsPrefsPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = MHS_PREFS_GET_PRIVATE (self);

  // 'placement new' to call the constructor for MhsPrefsPrivate
  new (reinterpret_cast<void *> (priv)) MhsPrefsPrivate;
  priv->observer.mPrefs = self;

  nsCOMPtr<nsIPrefService> pref_service
    = do_GetService (NS_PREFSERVICE_CONTRACTID, &rv);
  if (NS_SUCCEEDED (rv))
    {
      priv->pref_service = pref_service;
      NS_ADDREF (pref_service);
    }
  else
    g_warning ("Failed to retrieve preferences service object");

  priv->branches_by_id = g_hash_table_new (g_direct_hash, g_direct_equal);
  priv->ids_by_branch = g_hash_table_new (g_direct_hash, g_direct_equal);

  // Initialise the root branch
  mhs_prefs_get_branch (self, NULL, NULL, NULL);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object (connection,
                                           MHS_SERVICE_PREFS_PATH,
                                           G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }
}

static void
mhs_prefs_finalize (GObject *object)
{
  MhsPrefs *self = MHS_PREFS (object);
  MhsPrefsPrivate *priv = self->priv;
  GHashTableIter iter;
  gpointer key, value;

  // Save prefs
  mhs_prefs_save_pref_file (self, NULL, NULL);

  // Unref branches and get rid of hash-table
  g_hash_table_iter_init (&iter, priv->branches_by_id);
  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      nsIPrefBranch *branch = (nsIPrefBranch *)value;
      NS_RELEASE (branch);
    }
  g_hash_table_destroy (priv->branches_by_id);
  g_hash_table_destroy (priv->ids_by_branch);

  // Release reference on preferences service
  if (priv->pref_service)
    NS_RELEASE (priv->pref_service);

  /* Explicitly call the destructor for the private data (so that it
     destructs without trying to free the memory) */
  priv->~MhsPrefsPrivate ();

  G_OBJECT_CLASS (mhs_prefs_parent_class)->finalize (object);
}

MhsPrefs *
mhs_prefs_new (void)
{
  MhsPrefs *self = (MhsPrefs *)g_object_new (MHS_TYPE_PREFS, NULL);
  return self;
}

nsIPrefService *
mhs_prefs_get_service (MhsPrefs *prefs)
{
  return prefs->priv->pref_service;
}

static nsCOMPtr<nsIFile>
mhs_prefs_file_from_path (const gchar *path, GError **error)
{
  nsresult rv;
  nsCOMPtr<nsILocalFile> local_file =
    do_GetService ("@mozilla.org/file/local;1", &rv);

  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return NULL;
    }

  rv = local_file->InitWithPath (NS_ConvertUTF8toUTF16 (path));

  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return NULL;
    }

  nsCOMPtr<nsIFile> file = do_QueryInterface (local_file);

  return file;
}

static gboolean
mhs_prefs_read_user (MhsPrefs  *self,
                        const gchar  *path,
                        GError      **error)
{
  MhsPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsCOMPtr<nsIFile> ns_file = mhs_prefs_file_from_path (path,
                                                           error);

  if (ns_file)
    {
      nsresult rv = priv->pref_service->ReadUserPrefs (ns_file);
      if (NS_FAILED (rv))
        {
          mhs_error_set_from_nsresult (rv, error);
          return FALSE;
        }
      else
        return TRUE;
    }
  else
    return FALSE;
}

static gboolean
mhs_prefs_reset (MhsPrefs  *self,
                    GError      **error)
{
  MhsPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsresult rv = priv->pref_service->ResetPrefs ();
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }
  return TRUE;
}

static gboolean
mhs_prefs_reset_user (MhsPrefs  *self,
                         GError      **error)
{
  MhsPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsresult rv = priv->pref_service->ResetUserPrefs ();
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }
  else
    return TRUE;
}

static gboolean
mhs_prefs_save_pref_file (MhsPrefs  *self,
                          const gchar  *path,
                          GError      **error)
{
  MhsPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsCOMPtr<nsIFile> ns_file = nsnull;

  if (path && (path[0] != '\0'))
    ns_file = mhs_prefs_file_from_path (path,
                                        error);

  if (!path || (path[0] == '\0') || ns_file)
    {
      nsresult rv = priv->pref_service->SavePrefFile (ns_file);
      if (NS_FAILED (rv))
        {
          mhs_error_set_from_nsresult (rv, error);
          return FALSE;
        }
      else
        return TRUE;
    }
  else
    {
      mhs_error_set_from_nsresult (NS_ERROR_UNEXPECTED, error);
      return FALSE;
    }
}

static gint
mhs_prefs_get_new_id ()
{
  static gint id = 0;
  return id ++;
}

static gboolean
mhs_prefs_get_branch (MhsPrefs  *self,
                      const gchar  *root,
                      gint         *id,
                      GError      **error)
{
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  // Cache the root branch
  if ((!root) || (root[0] == '\0'))
    {
      branch = (nsIPrefBranch *)
        g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (0));
      if (branch)
        {
          NS_ADDREF (branch);
          if (id)
            *id = 0;
          return TRUE;
        }
    }

  nsresult rv = priv->pref_service->GetBranch (root, &branch);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  gint new_id = mhs_prefs_get_new_id ();
  g_hash_table_insert (priv->branches_by_id, GINT_TO_POINTER (new_id), branch);
  g_hash_table_insert (priv->ids_by_branch, branch, GINT_TO_POINTER (new_id));

  if (id)
    *id = new_id;

  return TRUE;
}

static gboolean
mhs_prefs_get_default_branch (MhsPrefs  *self,
                                 const gchar  *root,
                                 gint         *id,
                                 GError      **error)
{
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsresult rv = priv->pref_service->GetDefaultBranch (root, &branch);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  gint new_id = mhs_prefs_get_new_id ();
  g_hash_table_insert (priv->branches_by_id, GINT_TO_POINTER (new_id), branch);

  if (id)
    *id = new_id;

  return TRUE;
}

static gboolean
mhs_prefs_release_branch (MhsPrefs  *self,
                             gint          id,
                             GError      **error)
{
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  // We don't remove the root branch
  if (id == 0)
    return TRUE;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  g_hash_table_remove (priv->branches_by_id, GINT_TO_POINTER (id));
  g_hash_table_remove (priv->ids_by_branch, branch);
  NS_RELEASE (branch);

  return TRUE;
}

// nsPrefBranch accessor methods

static gboolean
mhs_prefs_branch_get_type (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gint         *type,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRInt32 value;
  rv = branch->GetPrefType (name, &value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (type)
    *type = value;

  return TRUE;
}

static gboolean
mhs_prefs_branch_get_bool (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gboolean     *value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRInt32 pr_value;
  rv = branch->GetBoolPref (name, &pr_value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (value)
    *value = pr_value;

  return TRUE;
}

static gboolean
mhs_prefs_branch_set_bool (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gboolean      value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->SetBoolPref (name, value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mhs_prefs_branch_get_int (MhsPrefs  *self,
                             gint          id,
                             const gchar  *name,
                             gint         *value,
                             GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRInt32 pr_value;
  rv = branch->GetIntPref (name, &pr_value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (value)
    *value = pr_value;

  return TRUE;
}

static gboolean
mhs_prefs_branch_set_int (MhsPrefs  *self,
                             gint          id,
                             const gchar  *name,
                             gint          value,
                             GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->SetIntPref (name, value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mhs_prefs_branch_get_char (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              char        **value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  char *ns_value;
  rv = branch->GetCharPref (name, &ns_value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (value)
    *value = g_strdup (ns_value);

  NS_Free (ns_value);

  return TRUE;
}

static gboolean
mhs_prefs_branch_set_char (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              const gchar  *value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->SetCharPref (name, value);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mhs_prefs_branch_has_user_value (MhsPrefs  *self,
                                    gint          id,
                                    const gchar  *name,
                                    gboolean     *result,
                                    GError      **error)
{
  nsresult rv;
  PRBool retval;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->PrefHasUserValue (name, &retval);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (result)
    *result = retval;

  return TRUE;
}

static gboolean
mhs_prefs_branch_lock (MhsPrefs  *self,
                          gint          id,
                          const gchar  *name,
                          GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->LockPref (name);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mhs_prefs_branch_is_locked (MhsPrefs  *self,
                               gint          id,
                               const gchar  *name,
                               gboolean     *result,
                               GError      **error)
{
  nsresult rv;
  PRBool retval;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->PrefIsLocked (name, &retval);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (result)
    *result = retval;

  return TRUE;
}

static gboolean
mhs_prefs_branch_unlock (MhsPrefs  *self,
                            gint          id,
                            const gchar  *name,
                            GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->UnlockPref (name);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mhs_prefs_branch_get_child_list (MhsPrefs   *self,
                                    gint           id,
                                    const gchar   *start,
                                    guint         *len,
                                    gchar       ***array,
                                    GError       **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRUint32 count;
  char **ns_array;
  rv = branch->GetChildList (start, &count, &ns_array);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (len)
    *len = count;
  if (array)
    {
      *array = (gchar **)g_malloc0 ((count + 1) * sizeof (gchar *));
      for (PRUint32 i = 0; i < count; i++)
        (*array)[i] = g_strdup (ns_array[i]);
    }

  NS_FREE_XPCOM_ALLOCATED_POINTER_ARRAY (count, ns_array);

  return TRUE;
}

static gboolean
mhs_prefs_branch_add_observer (MhsPrefs     *self,
                               gint          id,
                               const gchar  *domain,
                               GError      **error)
{
  nsresult rv;
  nsIPrefBranch2 *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch2 *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->AddObserver (domain,
                            static_cast<nsIObserver *>(&priv->observer),
                            PR_FALSE);
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mhs_prefs_branch_remove_observer (MhsPrefs     *self,
                                  gint          id,
                                  const gchar  *domain,
                                  GError      **error)
{
  nsresult rv;
  nsIPrefBranch2 *branch;
  MhsPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch2 *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mhs_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->RemoveObserver (domain,
                               static_cast<nsIObserver *>(&priv->observer));
  if (NS_FAILED (rv))
    {
      mhs_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

